import React from 'react';
import constant from 'lodash/constant';
import { mockResizeObserver, mountWithStoreAndTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { AnalyticsCards, AnalyticsCardsProps } from './analytics-cards.component';
import { ANALYTICS_SAMPLE_CARDS, NUMBER_OF_LOADING_CARD_SKELETONS } from '../constants';
import { CssCell, CssGrid, DotButton, DotDialog, DotEmptyFilterState, DotEmptyState } from '@digital-ai/dot-components';
import { AnalyticsCard } from './analytics-card.component';
import { AnalyticsCardSkeleton } from './analytics-card.skeleton';

describe('AnalyticsCards', () => {
    let wrapper: ReactWrapper;
    const onFavoriteDashboard = jest.fn();

    const defaultProps: AnalyticsCardsProps = {
        cards: ANALYTICS_SAMPLE_CARDS,
        isFavoritePage: false,
        isLoading: false,
        isSearchOrFilterApplied: false,
        isAnalyticsConfigured: false,
        onFavoriteDashboard,
        updatingDashboardIds: [],
    };

    const dispatch = jest.fn();

    const getAnalyticsCardsWrapper = () => wrapper.find('div.analytics-cards');
    const getCssGrid = () => wrapper.find(CssGrid);
    const getCssCells = () => wrapper.find(CssCell);
    const getAnalyticsCards = () => wrapper.find(AnalyticsCard);
    const getAnalyticsCardsSkeletons = () => wrapper.find(AnalyticsCardSkeleton);
    const getEmptyState = () => wrapper.find(DotEmptyState);
    const getEmptyFilterState = () => wrapper.find(DotEmptyFilterState);
    const getDotDialog = () => wrapper.find(DotDialog);

    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    const togglePendo = (enabled: boolean) => (window.pendo = { isSendingEvents: constant(enabled) });

    const mount = (props = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<AnalyticsCards {...props} />, dispatch);
    };

    beforeEach(() => {
        mockResizeObserver();
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render with correct elements', () => {
        mount();
        expect(getAnalyticsCardsWrapper()).toExist();
        expect(getCssGrid()).toExist();

        const cssCells = getCssCells();
        expect(cssCells.length).toBe(ANALYTICS_SAMPLE_CARDS.length);

        cssCells.forEach((cssCell: ReactWrapper<typeof CssCell>) => {
            expect(cssCell).toExist();
        });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.length).toBe(ANALYTICS_SAMPLE_CARDS.length);

        analyticsCards.forEach((analyticsCard, index) => {
            expect(analyticsCard).toExist();
            expect(analyticsCard.props().isFavorite).toBe(index === 5 || index === 6);
        });
    });

    it('should handle favorite toggle', () => {
        mount({ ...defaultProps, isFavoritePage: true });
        const analyticCard = getAnalyticsCards().at(0);
        analyticCard.invoke('onFavoriteClick')();
        expect(onFavoriteDashboard).toHaveBeenLastCalledWith('1', false);
    });

    it('should render empty state with correct title when on "Favorites" tab and no data is present', () => {
        mount({ ...defaultProps, cards: [], isFavoritePage: true });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.length).toBe(0);

        const emptyState = getEmptyState();
        expect(emptyState).toExist();
        expect(emptyState.props().title).toBe('No favorites have been added so far');
        expect(emptyState.props().imageSrc).toBe(undefined);
        expect(emptyState.props().illustrationId).toBe('digitalai-favorite');
    });

    it('should render empty state with correct title when on "All dashboards" tab and no data is present', () => {
        mount({ ...defaultProps, cards: [] });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.length).toBe(0);

        const emptyState = getEmptyState();
        expect(emptyState).toExist();
        expect(emptyState.props().title).toBe('No dashboards yet');
        expect(emptyState.props().imageSrc).toBe('static/@project.version@/styles/img/astronaut_rafiki.svg');
        expect(emptyState.props().illustrationId).toBe(undefined);
    });

    it('should render empty state with correct title when on "All dashboards" tab with analytics connection configured and no data is present', () => {
        mount({ ...defaultProps, cards: [], isAnalyticsConfigured: true });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.length).toBe(0);

        const emptyState = getEmptyState();
        expect(emptyState).toExist();
        expect(emptyState.props().title).toBe('Analytics dashboards');
        expect(emptyState.find(DotButton)).toExist();
        expect(emptyState.props().imageSrc).toBe('static/@project.version@/styles/img/data_analysis.svg');
        expect(emptyState.props().illustrationId).toBe(undefined);
    });

    it('should render empty state with correct title when search is applied and no data is present', () => {
        mount({ ...defaultProps, cards: [], isSearchOrFilterApplied: true });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.length).toBe(0);

        const emptyFilterState = getEmptyFilterState();
        expect(emptyFilterState).toExist();
        expect(emptyFilterState.text()).toContain('No results found');
    });

    it('should handle open click event when isAnalyticsConfigured is true', () => {
        mount({ ...defaultProps, isAnalyticsConfigured: true });
        const analyticsCard = getAnalyticsCards().at(0);
        analyticsCard.invoke('onOpenDashboardClick')();
        expect(window.location.href).toBe('http://localhost/#/analytics/1');
        expect(getDotDialog()).not.toExist();
    });

    it('should NOT open dialog window when isAnalyticsConfigured is false but pendo is enabled', () => {
        togglePendo(true);
        mount({ ...defaultProps, isAnalyticsConfigured: false });
        const analyticsCard = getAnalyticsCards().at(0);
        analyticsCard.invoke('onOpenDashboardClick')();
        expect(getDotDialog()).not.toExist();
    });

    it('should open dialog window when isAnalyticsConfigured is false and pendo is not enabled', () => {
        togglePendo(false);
        mount({ ...defaultProps, isAnalyticsConfigured: false });
        mount({ ...defaultProps, isAnalyticsConfigured: false });
        const analyticsCard = getAnalyticsCards().at(0);
        analyticsCard.invoke('onOpenDashboardClick')();
        const dotDialog = getDotDialog();
        expect(dotDialog).toExist();
        expect(dotDialog).toIncludeText(
            'You are seeing this message because your Release product setup has disabled guides and in-product analytics, or you are in an air-gapped environment. Please contact your Release administrator for further assistance.',
        );
        const dotDialogProps = dotDialog.props();
        expect(dotDialogProps.cancelButtonVisible).toBe(false);
        expect(dotDialogProps.closeIconVisible).toBe(true);
        expect(dotDialogProps.closeOnClickAway).toBe(false);
        expect(dotDialogProps.open).toBe(true);
        expect(dotDialogProps.submitButtonProps).toStrictEqual({
            label: 'Close',
        });
        expect(dotDialogProps.title).toBe('Content not available');

        dotDialog.invoke('onSubmit')?.('' as never);
        expect(getDotDialog()).not.toExist();
    });

    it('should handle open click event in a folder when isAnalyticsConfigured is true', () => {
        mount({ ...defaultProps, isAnalyticsConfigured: true, folderId: 'Applications/Folder1' });
        const analyticsCard = getAnalyticsCards().at(0);
        analyticsCard.invoke('onOpenDashboardClick')();
        expect(window.location.href).toBe('http://localhost/#/folders/Applications/Folder1/analytics/1');
    });

    it('should render correct elements when cards are present but isLoading is set to true', () => {
        mount({ ...defaultProps, isAnalyticsConfigured: true, isLoading: true });
        const skeletons = getAnalyticsCardsSkeletons();
        expect(skeletons.length).toBe(NUMBER_OF_LOADING_CARD_SKELETONS);
    });

    it('should have isUpdating flag raised on the AnalyticsCard components on which id is contained within updatingDashboardIds array', () => {
        mount({ ...defaultProps, isAnalyticsConfigured: true, updatingDashboardIds: ['1', '2'] });
        const cards = getAnalyticsCards();
        cards.forEach((card, index) => {
            expect(card.props().isUpdating).toBe(index === 0 || index === 1);
        });
    });
});
