package com.xebialabs.xlrelease.analytics.service

import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.analytics._
import com.xebialabs.xlrelease.planner.Planner.{AUTOMATED_TASK_DURATION, MANUAL_TASK_DURATION}
import com.xebialabs.xlrelease.repository.IdMatchers.{PhaseId, ReleaseId, TaskId}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.utils.DateVariableUtils.printDate
import com.xebialabs.xlrelease.utils.Graph
import com.xebialabs.xlrelease.variable.VariableHelper
import org.joda.time.{DateTime, Duration}

import java.util.Date
import scala.collection.mutable
import scala.jdk.CollectionConverters._

class PlanCalculator(variableResolver: VariableResolver) {
  private val computed: mutable.Map[String, ProjectedItem] = mutable.Map.empty[String, ProjectedItem]
  private var releaseMap: Map[String, Release] = Map.empty
  private val now = DateTime.now()

  def calculateDates(releases: List[Release]): mutable.Map[String, ProjectedItem] = {
    releaseMap = releases.map(r => (r.getId, r)).toMap

    releases.foreach(calculateReleaseDates(_, now))
    fixupDependencyDetails()
    computed
  }

  private def calculateStartDate(planItem: PlanItem, projectedStartDate: DateTime): DateTime = {
    val actualStart = optionalDateTime(planItem.getStartDate)
    actualStart match {
      case Some(start) => start
      case None => {
        optionalDateTime(planItem.getScheduledStartDate) match {
          case Some(scheduled) => if (scheduled.isBefore(projectedStartDate)) projectedStartDate else scheduled
          case None => projectedStartDate
        }
      }
    }
  }

  private def calculateTaskDuration(task: Task): Duration = {
    if (task.hasPlannedDuration) {
      Duration.standardSeconds(task.getPlannedDuration.longValue)
    }
    else if (task.isAutomated) {
      Duration.standardMinutes(AUTOMATED_TASK_DURATION)
    }
    else {
      Duration.standardMinutes(MANUAL_TASK_DURATION)
    }
  }

  private def calculateEndDate(planItem: PlanItem, startDate: DateTime, projectedEndDate: DateTime): DateTime = {
    val actualEnd = optionalDateTime(planItem.getEndDate)
    actualEnd match {
      case Some(end) => end
      case None => {
        var end = optionalDateTime(planItem.getDueDate) match {
          case Some(due) => due
          case None => projectedEndDate
        }
        end = latest(startDate, end)
        if (planItem.isActive && end.isBefore(now)) now else end
      }
    }
  }

  private def calculateReleaseDates(release: Release, now: DateTime): ProjectedRelease = {
    computed.getOrElse(release.getId, {
      val releaseVars = VariableResolver.buildVarMap(release.getVariables.asScala.toSeq)

      val result = resultContainer(release, new ProjectedRelease, releaseVars)

      val scheduled = optionalDateTime(release.getScheduledStartDate)
      val earliestPossibleStart = if (scheduled.isDefined && now.isBefore(scheduled.get)) scheduled.get else now

      val lastPhaseEnd = release.getPhases.asScala.foldLeft(earliestPossibleStart) { (nextEarliestStart, phase) =>
        val calculated = calculatePhaseDates(phase, nextEarliestStart, releaseVars)
        result.phases.add(calculated.result)
        calculated.latestEndDate
      }
      val start = calculateStartDate(release, getFirstChildStart(earliestPossibleStart, result))
      val end = optionalDateTime(release.getEndDate).getOrElse(lastPhaseEnd)
      fillDates(result, start, end)
      result
    }).asInstanceOf[ProjectedRelease]
  }

  private def calculatePhaseDates(phase: Phase, earliestStartDate: DateTime, releaseVars: Map[String, String]): ResultAndLatestEnd[ProjectedPhase] = {
    val result = resultContainer(phase, new ProjectedPhase, releaseVars)

    val lastTaskEnd = phase.getTasks.asScala.foldLeft(earliestStartDate) { (nextEarliestStart, task) =>
      val calculated = calculateTaskDates(task, nextEarliestStart, releaseVars)
      result.tasks.add(calculated.result)
      calculated.latestEndDate
    }

    val start = calculateStartDate(phase, getFirstChildStart(earliestStartDate, result))
    val end = optionalDateTime(phase.getEndDate).getOrElse(lastTaskEnd)
    fillDates(result, start, end)
    ResultAndLatestEnd(result, end)
  }

  private def calculateTaskDates(task: Task, earliestStartDate: DateTime, releaseVars: Map[String, String]): ResultAndLatestEnd[ProjectedTask] = {
    task match {
      case p: ParallelGroup => calculateParallelGroupDates(p, earliestStartDate, releaseVars)
      case s: SequentialGroup => calculateSequentialGroupDates(s, earliestStartDate, releaseVars)
      case g: GateTask => calculateGateTaskDates(g, earliestStartDate, releaseVars)
      case t: Task =>
        val result = resultContainer(t, new ProjectedTask, releaseVars)
        val start = calculateStartDate(t, earliestStartDate)
        val end = calculateEndDate(t, start, start.plus(calculateTaskDuration(t)))
        fillDates(result, start, end)

        if ((task.isCompletedInAdvance || task.isSkippedInAdvance) && end.isBefore(earliestStartDate)) {
          ResultAndLatestEnd(result, earliestStartDate)
        } else {
          ResultAndLatestEnd(result, end)
        }
    }
  }

  private case class StartAndEnd(start: DateTime, end: DateTime)

  private def calculateParallelGroupDates(group: ParallelGroup, earliestStartDate: DateTime, releaseVars: Map[String, String]): ResultAndLatestEnd[ProjectedTask] = {
    val result = resultContainer(group, new ProjectedTaskGroup, releaseVars)

    val graph = Graph(group.getLinks.asScala.map(link => Graph.Edge(link.getSource.getId -> link.getTarget.getId)))
    val lastSubTaskEnd = if (!graph.hasCycle) {
      val subtasks = group.getTasks.asScala.map(task => (task.getId, task)).toMap
      val order = (subtasks.keySet diff graph.nodes).toList ++ graph.order
      order.foreach { id =>
        val subtask = subtasks(id)
        val nextEarliestStart = findMaxDate(earliestStartDate, graph.incoming(id).toList.flatMap(computed.get).map(c => new DateTime(c.endDate)))
        result.tasks.add(calculateTaskDates(subtask, nextEarliestStart, releaseVars).result)
      }

      val bestDates = subtasks.keySet.toList.flatMap(computed.get)
        .foldLeft(StartAndEnd(earliestStartDate, earliestStartDate)) {
          (startAndEnd, computedSubTask) =>
            val newStart = earliest(startAndEnd.start, new DateTime(computedSubTask.startDate))
            val newEnd = latest(startAndEnd.end, new DateTime(computedSubTask.endDate))
            StartAndEnd(newStart, newEnd)
        }

      // Make sure parallel group picks the later of the earliest possible start for the parallel group
      // and the best start date from the sub tasks.  It is possible the subtasks were completed in advance
      // and have start dates that don't make sense for projecting the start of the group.
      val start = calculateStartDate(group, latest(bestDates.start, earliestStartDate))
      val end = optionalDateTime(group.getEndDate).getOrElse(bestDates.end)
      fillDates(result, start, end)
      end
    } else {
      fillDates(result, earliestStartDate, earliestStartDate)
      earliestStartDate
    }
    ResultAndLatestEnd(result, lastSubTaskEnd)
  }

  private def calculateSequentialGroupDates(group: SequentialGroup, earliestStartDate: DateTime, releaseVars: Map[String, String]): ResultAndLatestEnd[ProjectedTask] = {
    val result = resultContainer(group, new ProjectedTaskGroup, releaseVars)

    val lastTaskEnd = group.getTasks.asScala.foldLeft(earliestStartDate) { (nextEarliestStart, task) =>
      val calculated = calculateTaskDates(task, nextEarliestStart, releaseVars)
      result.tasks.add(calculated.result)
      calculated.latestEndDate
    }

    val start = calculateStartDate(group, getFirstChildStart(earliestStartDate, result))
    val end = optionalDateTime(group.getEndDate).getOrElse(lastTaskEnd)
    fillDates(result, start, end)
    ResultAndLatestEnd(result, end)
  }

  private def calculateDependencyEndDate(id: String, gateStart: DateTime): DateTime = {
    // Check if dependent item has already been computed
    computed.get(id) match {
      case Some(result) => new DateTime(result.endDate) // It has been computed so use its end date
      case None => { // Not computed yet, attempt to calculate dates
        val depReleaseId = try {
          Option(Ids.releaseIdFrom(id))
        } catch {
          case _: Throwable => None
        }

        if (depReleaseId.isDefined && releaseMap.contains(depReleaseId.get)) {
          // the dependent release hasn't been calculated yet, so calculate now
          calculateReleaseDates(releaseMap(depReleaseId.get), now)
          // If the id still isn't in the computed map then there must be a circular reference the
          // prevent the full graph from being walked, just use the start date of the gate
          computed.get(id) match {
            case Some(result) => new DateTime(result.endDate)
            case None => gateStart
          }
        } else {
          // The dependency isn't one of the releases we are processing, likely has been moved to the archive
          // so the dependency has already been met
          gateStart
        }
      }
    }
  }

  private def calculateGateTaskDates(gate: GateTask, earliestStartDate: DateTime, releaseVars: Map[String, String]): ResultAndLatestEnd[ProjectedTask] = {
    val result = resultContainer(gate, new ProjectedGateTask, releaseVars)
    result.dependencies = transformDependencies(gate.getDependencies.asScala.toSeq).asJava

    val start = calculateStartDate(gate, earliestStartDate)

    val end = optionalDateTime(gate.getEndDate).getOrElse {
      val dependencyEndDates = gate.getDependencies.asScala
        .map(_.getTargetId)
        .filter(id => !VariableHelper.containsVariables(id))
        .map(id => calculateDependencyEndDate(id, start))
        .toList

      if (dependencyEndDates.nonEmpty) {
        val latestDependency = findMaxDate(start, dependencyEndDates)
        // Circular dependencies may result in an active gate getting end dates before now, swap out for now since that doesn't make sense.
        if (gate.isActive && latestDependency.isBefore(now)) now else latestDependency
      } else {
        calculateEndDate(gate, start, start.plus(calculateTaskDuration(gate)))
      }
    }

    fillDates(result, start, end)
    if ((gate.isCompletedInAdvance || gate.isSkippedInAdvance) && end.isBefore(earliestStartDate)) {
      ResultAndLatestEnd(result, earliestStartDate)
    } else {
      ResultAndLatestEnd(result, end)
    }
  }

  private def resolveVariables(input: String, folderId: String, releaseVars: Map[String, String]): String = {
    if (variableResolver != null && VariableHelper.containsVariables(input)) {
      val vars = VariableHelper.collectVariables(input).asScala
        .map(v => (v, variableResolver.getValue(VariableHelper.withoutVariableSyntax(v), folderId, releaseVars).getOrElse(null)))
        .filter(v => v._2 != null).toMap

      VariableHelper.replaceAll(input, vars.asJava)
    } else {
      input
    }
  }

  private def resultContainer[T <: ProjectedItem](planItem: PlanItem, result: T, releaseVars: Map[String, String]): T = {
    computed.put(planItem.getId, result)
    result.id = planItem.getId
    result.title = resolveVariables(planItem.getTitle, Ids.findFolderId(planItem.getId), releaseVars)
    result.`type` = planItem.getType.toString
    planItem match {
      case r: Release => result.status = r.getStatus.toString
      case p: Phase => result.status = p.getStatus.toString
      case t: Task => result.status = t.getStatus.toString
    }
    result
  }

  private def fillDates(target: ProjectedItem, start: DateTime, end: DateTime): Unit = {
    target.startDate = start.toDate
    target.startDateString = printDate(target.startDate)
    target.endDate = end.toDate
    target.endDateString = printDate(target.endDate)
  }

  private def optionalDateTime(date: Date): Option[DateTime] = Option(date).map(d => new DateTime(d))

  private def latest(a: DateTime, b: DateTime): DateTime = if (a.isAfter(b)) a else b

  private def earliest(a: DateTime, b: DateTime): DateTime = if (a.isBefore(b)) a else b

  private def findMaxDate(default: DateTime, dates: List[DateTime]): DateTime = dates.foldLeft(default)(latest)

  private def findMinDate(default: DateTime, dates: List[DateTime]): DateTime = dates.foldLeft(default)(earliest)

  private case class ResultAndLatestEnd[T <: ProjectedItem](result: T, latestEndDate: DateTime)

  private def getFirstChildStart(default: DateTime, parentResult: ProjectedItem): DateTime = {
    val childStart = parentResult match {
      case r: ProjectedRelease => if (r.phases.isEmpty) default else new DateTime(r.phases.get(0).startDate)
      case p: ProjectedPhase => if (p.tasks.isEmpty) default else new DateTime(p.tasks.get(0).startDate)
      case tg: ProjectedTaskGroup => if (tg.tasks.isEmpty) default else new DateTime(tg.tasks.get(0).startDate)
      case _ => default
    }
    if (childStart.isBefore(default)) default else childStart
  }

  private val dependencyMap: mutable.Map[String, ProjectedDependency] = mutable.Map.empty[String, ProjectedDependency]

  private def transformDependency(dependency: Dependency): ProjectedDependency = {
    val id = dependency.getTargetId

    if (VariableHelper.containsVariables(id)) {
      // Don't stick in the dependency map, the variable is not resolvable so no way to update later
      val dep = new ProjectedDependency
      dep.targetId = "unknown"
      dep.targetType = "unknown"
      dep.targetTitle = "unknown"
      dep.variable = id
      dep
    } else {
      dependencyMap.getOrElse(id, {
        val dep = new ProjectedDependency
        dep.targetId = id
        dep.targetTitle = Option(dependency.getArchivedTargetTitle) match {
          case Some(title) => title
          case None => null
        }
        dependencyMap.put(id, dep)
        dep
      })
    }
  }

  private def transformDependencies(dependencies: Seq[Dependency]): Seq[ProjectedDependency] = {
    dependencies.filter(_.getTargetId != null).map(transformDependency)
  }

  case class IdAndTitle(id: String, title: Option[String])

  private def getComputedTitle(id: String): Option[String] = {
    val maybeTarget = computed.get(id)
    if (maybeTarget.isDefined) Some(maybeTarget.get.title) else None
  }

  private def findReleaseId(id: String): Option[String] = {
    try {
      Option(Ids.releaseIdFrom(id))
    } catch {
      case _: Throwable => None
    }
  }

  private def joinTitles(parentTitle: Option[String], childTitle: Option[String]): Option[String] = {
    parentTitle.map(pTitle => {
      childTitle match {
        case Some(cTitle) => s"${pTitle} / ${cTitle}"
        case None => pTitle
      }
    })
  }

  private def dependencyTitle(id: String): Option[String] = {
    findReleaseId(id)
      .map(relId => IdAndTitle(relId, getComputedTitle(relId)))
      .map(releaseInfo => {
        val subIds = id.substring(releaseInfo.id.length).split("/").filter(subId => subId != "")
        subIds.foldLeft(releaseInfo) { (parentInfo, childId) =>
          val fullChildId = s"${parentInfo.id}/$childId"
          val childTitle = joinTitles(parentInfo.title, getComputedTitle(fullChildId))
          IdAndTitle(fullChildId, childTitle)
        }
      })
      .flatMap(_.title)
  }

  private def fixupDependencyDetails(): Unit = {
    dependencyMap.values.foreach(dep => {
      dep.targetType = dep.targetId match {
        case ReleaseId(_) => "release"
        case PhaseId(_) => "phase"
        case TaskId(_) => "task"
        case _ => "unknown"
      }

      if (dep.targetTitle == null) {
        dep.targetTitle = dependencyTitle(dep.targetId) match {
          case Some(title) => title
          case None => "unknown"
        }
      }
    })
  }
}
