package com.xebialabs.xlrelease.analytics.service

import com.xebialabs.xlrelease.analytics.service.VariableResolver.buildVarMap
import com.xebialabs.xlrelease.api.v1.{ConfigurationApi, FolderApi}
import com.xebialabs.xlrelease.domain.variables.{StringVariable, Variable}
import com.xebialabs.xlrelease.variable.VariableHelper

import scala.collection.mutable
import scala.jdk.CollectionConverters._

object VariableResolver {
  def buildVarMap(variables: Seq[Variable]): Map[String, String] = {
    variables
      .filter(_ match {
        case stringVar: StringVariable => {
          val v = stringVar.getValue
          v != null && !v.isEmpty
        }
        case _ => false
      })
      .map(variable => (variable.getKey, variable.getValueAsString)).toMap
  }
}

class VariableResolver(configurationApi: ConfigurationApi, folderApi: FolderApi) {
  private var loadedGlobal: Boolean = false
  private var globalVars: Map[String, String] = _
  private val folderVars: mutable.Map[String, Map[String, String]] = mutable.Map.empty[String, Map[String, String]]

  private def loadGlobalVariables(): Map[String, String] = {
    if (!loadedGlobal) {
      globalVars = buildVarMap(configurationApi.getGlobalVariables().asScala.toSeq)
      loadedGlobal = true
    }
    globalVars
  }

  private def loadFolderVariables(folderId: String): Map[String, String] = {
    if(!folderVars.contains(folderId)) {
      folderVars.put(folderId, buildVarMap(folderApi.listVariables(folderId, true).asScala.toSeq))
    }
    folderVars(folderId)
  }

  def getValue(name: String, folderId: String, releaseVars: Map[String, String]): Option[String] = {
    if (VariableHelper.isGlobalVariable(name)) {
      loadGlobalVariables().get(name)
    } else if (VariableHelper.isFolderVariable(name)) {
      loadFolderVariables(folderId).get(name)
    } else {
      releaseVars.get(name)
    }
  }
}
