import { analytics, AnalyticsTabsEnum, initialState } from './analytics.reducer';
import { ANALYTICS_CATEGORIES, DIGITAL_AI_AUTHOR } from '../constants';
import { DEFAULT_PAGINATION } from '../../../../../../../../../core/xlr-ui/app/constants/pagination';
import { AnalyticsCardData } from '../types';

describe('analytics reducer', () => {
    const defaultPagination = DEFAULT_PAGINATION;
    const {
        addToUpdatingDashboardIds,
        clearSelectedDashboard,
        favoriteDashboardFromDetails,
        favoriteDashboardFromList,
        filterCards,
        init,
        loadDashboard,
        loadFavoriteCards,
        refreshToken,
        removeFromFavoritesCards,
        removeFromUpdatingDashboardIds,
        replaceCardInAllCards,
        resetAnalytics,
        setAllCards,
        setAuthors,
        setCardSearch,
        setCategories,
        setFavoriteCards,
        setIntelligenceConfiguration,
        setIsCategoriesLoading,
        setIsDrawerOpened,
        setIsLoading,
        setManageDashboardUrl,
        setSelectedDashboard,
        setSelectedTab,
        setTokenData,
        setTotalAvailableCards,
        setTotalAvailableFavoriteCards,
        setTotalPages,
        setUpdatingDashboardIds,
    } = analytics.actions;

    it('should have the initial state', () => {
        expect(analytics.name).toBe('analytics');
        expect(analytics.reducer(undefined, { type: 'default' })).toBe(initialState);
        expect(Object.keys(analytics.actions)).toStrictEqual([
            'init',
            'filterCards',
            'setAllCards',
            'setFavoriteCards',
            'setCategories',
            'setAuthors',
            'setIsDrawerOpened',
            'setIntelligenceConfiguration',
            'setCardSearch',
            'setIsCategoriesLoading',
            'setIsLoading',
            'setTotalPages',
            'setTotalAvailableCards',
            'setTotalAvailableFavoriteCards',
            'loadDashboard',
            'loadFavoriteCards',
            'favoriteDashboardFromList',
            'favoriteDashboardFromDetails',
            'setSelectedDashboard',
            'setSelectedTab',
            'clearSelectedDashboard',
            'setManageDashboardUrl',
            'refreshToken',
            'setTokenData',
            'setUpdatingDashboardIds',
            'addToUpdatingDashboardIds',
            'removeFromUpdatingDashboardIds',
            'replaceCardInAllCards',
            'removeFromFavoritesCards',
            'resetAnalytics',
        ]);
    });

    describe('reducers', () => {
        describe('init', () => {
            it('should NOT update state', () => {
                const state = analytics.reducer(initialState, init({ name: 'abc', authorNames: ['bob'] }));
                expect(state).toStrictEqual(initialState);
            });
        });
        describe('filterCards', () => {
            it('should not update state', () => {
                const state = analytics.reducer(initialState, filterCards({ name: 'abc', authorNames: ['bob'], ...defaultPagination }));
                expect(state).toStrictEqual(initialState);
            });
        });
        describe('setAllCards', () => {
            it('should update state', () => {
                const cards = [
                    {
                        contentText: 'Website Traffic Report for October',
                        id: '1',
                        hasFolderFilter: false,
                        embedConfig: {} as never,
                        externalId: 1,
                        embeddingId: 'asdasdsad',
                        imageSrc: 'test',
                        isFavorite: true,
                        serverUrl: 'http',
                        subtitle: DIGITAL_AI_AUTHOR,
                        tags: [ANALYTICS_CATEGORIES[0], ANALYTICS_CATEGORIES[1], ANALYTICS_CATEGORIES[2]],
                        title: 'Onboarding plan and progress A',
                    },
                ];
                const state = analytics.reducer(initialState, setAllCards(cards));
                expect(state.allCards).toStrictEqual(cards);
            });
        });
        describe('setFavoriteCards', () => {
            it('should update state', () => {
                const cards = [
                    {
                        contentText: 'Favorite card',
                        id: '1',
                        hasFolderFilter: false,
                        embedConfig: {} as never,
                        externalId: 1,
                        embeddingId: 'asdasdsad',
                        imageSrc: 'test',
                        isFavorite: true,
                        serverUrl: 'http',
                        subtitle: 'Digital.ai',
                        tags: [ANALYTICS_CATEGORIES[0], ANALYTICS_CATEGORIES[1], ANALYTICS_CATEGORIES[2]],
                        title: 'Onboarding plan and progress A',
                    },
                ];
                const state = analytics.reducer(initialState, setFavoriteCards(cards));
                expect(state.favoriteCards).toStrictEqual(cards);
            });
        });
        describe('setCategories', () => {
            it('should update state', () => {
                const categories = ['cat1'];
                const state = analytics.reducer(initialState, setCategories(categories));
                expect(state.categories).toStrictEqual(categories);
            });
        });
        describe('setAuthors', () => {
            it('should update state', () => {
                const authors = ['bob', 'john'];
                const state = analytics.reducer(initialState, setAuthors(authors));
                expect(state.authors).toStrictEqual(authors);
            });
        });
        describe('setIsDrawerOpened', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setIsDrawerOpened(true));
                expect(state.isDrawerOpened).toBe(true);
            });
        });

        describe('setIntelligenceConfiguration', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setIntelligenceConfiguration({} as never));
                expect(state.intelligenceConfiguration).toStrictEqual({});
            });
        });

        describe('setCardSearch', () => {
            it('should update state', () => {
                const cardSearch = { name: 'abc', authorNames: ['bob'], categories: ['test'], ...defaultPagination };
                const state = analytics.reducer(initialState, setCardSearch(cardSearch));
                expect(state.cardSearch).toStrictEqual(cardSearch);
            });
        });

        describe('setIsCategoriesLoading', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setIsCategoriesLoading(true));
                expect(state.isCategoriesLoading).toStrictEqual(true);
            });
        });
        describe('setIsLoading', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setIsLoading(true));
                expect(state.isLoading).toStrictEqual(true);
            });
        });
        describe('setTotalPages', () => {
            it('should update state', () => {
                const totalPages = 10;
                const state = analytics.reducer(initialState, setTotalPages(totalPages));
                expect(state.totalPages).toBe(totalPages);
            });
        });
        describe('setTotalAvailableCards', () => {
            it('should update state', () => {
                const totalAvailableCards = 3;
                const state = analytics.reducer(initialState, setTotalAvailableCards(totalAvailableCards));
                expect(state.totalAvailableCards).toBe(totalAvailableCards);
            });
        });
        describe('setTotalAvailableFavoriteCards', () => {
            it('should update state', () => {
                const totalAvailableFavoriteCards = 4;
                const state = analytics.reducer(initialState, setTotalAvailableFavoriteCards(totalAvailableFavoriteCards));
                expect(state.totalAvailableFavoriteCards).toBe(totalAvailableFavoriteCards);
            });
        });
        describe('loadDashboard', () => {
            it('should not update state', () => {
                const state = analytics.reducer(initialState, loadDashboard('222'));
                expect(state).toStrictEqual(initialState);
            });
        });
        describe('loadFavoriteCards', () => {
            it('should not update state', () => {
                const state = analytics.reducer(initialState, loadFavoriteCards());
                expect(state).toStrictEqual(initialState);
            });
        });
        describe('favoriteDashboardFromDetails', () => {
            it('should not update state', () => {
                const state = analytics.reducer(initialState, favoriteDashboardFromDetails({ id: '123', isUnfavorite: false }));
                expect(state).toStrictEqual(initialState);
            });
        });
        describe('favoriteDashboardFromList', () => {
            it('should not update state', () => {
                const state = analytics.reducer(initialState, favoriteDashboardFromList({ id: '222', isUnfavorite: true }));
                expect(state).toStrictEqual(initialState);
            });
        });
        describe('setSelectedDashboard', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setSelectedDashboard({} as never));
                expect(state.selectedDashboard).toStrictEqual({});
            });
        });
        describe('setSelectedTab', () => {
            it('should update state', () => {
                const state = analytics.reducer(
                    { ...initialState, cardSearch: { ...initialState.cardSearch, page: 3 }, selectedTab: AnalyticsTabsEnum.AllDashboards },
                    setSelectedTab(AnalyticsTabsEnum.Favorites),
                );
                expect(state.selectedTab).toBe(AnalyticsTabsEnum.Favorites);
                expect(state.cardSearch.page).toBe(0);
            });
        });
        describe('clearSelectedDashboard', () => {
            it('should update state', () => {
                const state = analytics.reducer({ ...initialState, selectedDashboard: 'a' } as never, clearSelectedDashboard());
                expect(state.selectedDashboard).toBeUndefined();
            });
        });
        describe('setManageDashboardUrl', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setManageDashboardUrl('url'));
                expect(state.manageDashboardUrl).toBe('url');
            });
        });
        describe('refreshToken', () => {
            it('should not update state', () => {
                const state = analytics.reducer(initialState, refreshToken());
                expect(state).toBe(initialState);
            });
        });
        describe('setTokenData', () => {
            it('should update state', () => {
                const state = analytics.reducer(initialState, setTokenData({ tokenExpiresAt: 1, tokenValue: 'token' }));
                expect(state.tokenExpiresAt).toBe(1);
                expect(state.tokenValue).toBe('token');
            });
        });
        describe('setUpdatingDashboardIds', () => {
            it('should update state', () => {
                const updatingDashboardIds = ['1', '2'];
                const state = analytics.reducer(initialState, setUpdatingDashboardIds(updatingDashboardIds));
                expect(state.updatingDashboardIds).toBe(updatingDashboardIds);
            });
        });
        describe('addToUpdatingDashboardIds', () => {
            it('should update state', () => {
                const state = analytics.reducer({ ...initialState, updatingDashboardIds: ['1', '2'] }, addToUpdatingDashboardIds('3'));
                expect(state.updatingDashboardIds).toStrictEqual(['1', '2', '3']);
            });
        });
        describe('removeFromUpdatingDashboardIds', () => {
            it('should update state', () => {
                const state = analytics.reducer({ ...initialState, updatingDashboardIds: ['1', '2'] }, removeFromUpdatingDashboardIds('1'));
                expect(state.updatingDashboardIds).toStrictEqual(['2']);
            });
        });
        describe('replaceCardInAllCards', () => {
            it('should update state', () => {
                const allCards: AnalyticsCardData[] = [
                    { id: '1', isFavorite: false },
                    { id: '2', isFavorite: false },
                    { id: '3', isFavorite: false },
                ] as never;
                const state = analytics.reducer({ ...initialState, allCards }, replaceCardInAllCards({ id: '2', isFavorite: true } as never));
                expect(state.allCards).toStrictEqual([
                    { id: '1', isFavorite: false },
                    { id: '2', isFavorite: true },
                    { id: '3', isFavorite: false },
                ]);
            });
        });
        describe('removeFromFavoritesCards', () => {
            it('should update state', () => {
                const state = analytics.reducer(
                    { ...initialState, favoriteCards: [{ id: '1' }, { id: '2' }] as never, totalAvailableFavoriteCards: 2 },
                    removeFromFavoritesCards('1'),
                );
                expect(state.favoriteCards).toStrictEqual([{ id: '2' }]);
                expect(state.totalAvailableFavoriteCards).toStrictEqual(1);
            });
        });
        describe('resetAnalytics', () => {
            it('should update state', () => {
                const state = analytics.reducer(
                    {
                        ...initialState,
                        selectedTab: AnalyticsTabsEnum.Favorites,
                        isDrawerOpened: true,
                        allCards: [{ id: '1' }, { id: '2' }, { id: '3' }] as never,
                        favoriteCards: [{ id: '1' }, { id: '2' }] as never,
                    },
                    resetAnalytics(),
                );
                expect(state).toStrictEqual({ ...initialState, selectedTab: AnalyticsTabsEnum.Favorites });
            });
        });
    });
});
