import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import noop from 'lodash/noop';
import { RootState } from '../../../../../../../../../core/xlr-ui/app/js/store.types';
import { AnalyticsCardData, IntelligenceHttpConnection } from '../types';
import { Pagination } from '../../../../../../../../../core/xlr-ui/app/types';
import { DEFAULT_PAGINATION } from '../../../../../../../../../core/xlr-ui/app/constants/pagination';
import { calculateLastLoadedPage, getUpdatedCards } from '../helpers';

export interface AnalyticsPageFilter {
    authorNames?: string[];
    categories?: string[];
    isFavorite?: boolean;
    name?: string;
}

export type CardSearch = AnalyticsPageFilter & Pagination;

export interface FavoriteDashboardAction {
    id: string;
    isUnfavorite: boolean;
}

export enum AnalyticsTabsEnum {
    AllDashboards,
    Favorites,
}

export interface AnalyticsState {
    allCards: AnalyticsCardData[];
    authors: string[];
    cardSearch: CardSearch;
    categories: string[];
    favoriteCards: AnalyticsCardData[];
    intelligenceConfiguration?: IntelligenceHttpConnection;
    isCategoriesLoading: boolean;
    isDrawerOpened: boolean;
    isLoading: boolean;
    manageDashboardUrl?: string;
    selectedDashboard?: AnalyticsCardData;
    selectedTab: AnalyticsTabsEnum;
    tokenExpiresAt?: number; // seconds in UTC
    tokenValue?: string;
    totalAvailableCards: number;
    totalAvailableFavoriteCards: number;
    totalPages: number;
    updatingDashboardIds: string[];
}

export interface SetTokenDataAction {
    tokenExpiresAt: number;
    tokenValue: string;
}

export const initialState: AnalyticsState = {
    allCards: [],
    authors: [],
    cardSearch: {
        ...DEFAULT_PAGINATION,
        authorNames: [],
        categories: [],
        isFavorite: false,
        name: '',
    },
    categories: [],
    favoriteCards: [],
    isCategoriesLoading: false,
    isDrawerOpened: false,
    isLoading: true,
    selectedTab: AnalyticsTabsEnum.AllDashboards,
    totalAvailableCards: 0,
    totalAvailableFavoriteCards: 0,
    totalPages: 0,
    updatingDashboardIds: [],
};

export const analytics = createSlice({
    name: 'analytics',
    initialState,
    reducers: {
        init: (_state, _action: PayloadAction<Partial<CardSearch> | undefined>) => noop(),
        filterCards: (_state, _action: PayloadAction<CardSearch>) => noop(),
        setAllCards: (state, action: PayloadAction<AnalyticsCardData[]>) => {
            state.allCards = action.payload;
        },
        setFavoriteCards: (state, action: PayloadAction<AnalyticsCardData[]>) => {
            state.favoriteCards = action.payload;
        },
        setCategories: (state, action: PayloadAction<string[]>) => {
            state.categories = action.payload;
        },
        setAuthors: (state, action: PayloadAction<string[]>) => {
            state.authors = action.payload;
        },
        setIsDrawerOpened: (state, action: PayloadAction<boolean>) => {
            state.isDrawerOpened = action.payload;
        },
        setIntelligenceConfiguration: (state, action: PayloadAction<IntelligenceHttpConnection | undefined>) => {
            state.intelligenceConfiguration = action.payload;
        },
        setCardSearch: (state, action: PayloadAction<CardSearch>) => {
            state.cardSearch = action.payload;
        },
        setIsCategoriesLoading: (state, action: PayloadAction<boolean>) => {
            state.isCategoriesLoading = action.payload;
        },
        setIsLoading: (state, action: PayloadAction<boolean>) => {
            state.isLoading = action.payload;
        },
        setTotalPages: (state, action: PayloadAction<number>) => {
            state.totalPages = action.payload;
        },
        setTotalAvailableCards: (state, action: PayloadAction<number>) => {
            state.totalAvailableCards = action.payload;
        },
        setTotalAvailableFavoriteCards: (state, action: PayloadAction<number>) => {
            state.totalAvailableFavoriteCards = action.payload;
        },
        loadDashboard: (_state, _action: PayloadAction<string>) => noop(),
        loadFavoriteCards: (_state) => noop(),
        favoriteDashboardFromList: (_state, _action: PayloadAction<FavoriteDashboardAction>) => noop(),
        favoriteDashboardFromDetails: (_state, _action: PayloadAction<FavoriteDashboardAction>) => noop(),
        setSelectedDashboard: (state, action: PayloadAction<AnalyticsCardData>) => {
            state.selectedDashboard = action.payload;
        },
        setSelectedTab: (state, action: PayloadAction<AnalyticsTabsEnum>) => {
            state.selectedTab = action.payload;
            state.cardSearch.page = calculateLastLoadedPage(
                action.payload === AnalyticsTabsEnum.AllDashboards ? state.allCards.length : state.favoriteCards.length,
                state.cardSearch.itemsPerPage,
            );
        },
        clearSelectedDashboard: (state) => {
            state.selectedDashboard = undefined;
        },
        setManageDashboardUrl: (state, action: PayloadAction<string>) => {
            state.manageDashboardUrl = action.payload;
        },
        setTokenData: (state, action: PayloadAction<SetTokenDataAction>) => {
            const { tokenValue, tokenExpiresAt } = action.payload;
            state.tokenValue = tokenValue;
            state.tokenExpiresAt = tokenExpiresAt;
        },
        setUpdatingDashboardIds: (state, action: PayloadAction<string[]>) => {
            state.updatingDashboardIds = action.payload;
        },
        addToUpdatingDashboardIds: (state, action: PayloadAction<string>) => {
            state.updatingDashboardIds.push(action.payload);
        },
        removeFromUpdatingDashboardIds: (state, action: PayloadAction<string>) => {
            state.updatingDashboardIds = state.updatingDashboardIds.filter((dashboardId) => dashboardId !== action.payload);
        },
        replaceCardInAllCards: (state, action: PayloadAction<AnalyticsCardData>) => {
            state.allCards = getUpdatedCards(action.payload, state.allCards);
        },
        removeFromFavoritesCards: (state, action: PayloadAction<string>) => {
            state.favoriteCards = state.favoriteCards.filter((card) => card.id !== action.payload);
            state.totalAvailableFavoriteCards = state.totalAvailableFavoriteCards - 1;
        },
        resetAnalytics: (state) => {
            return { ...initialState, selectedTab: state.selectedTab };
        },
    },
});

export const getAnalyticsState = (state: RootState): AnalyticsState => state.analytics;
export const getAnalyticsStateDashboard = (state: RootState): AnalyticsCardData | undefined => getAnalyticsState(state).selectedDashboard;
