import React from 'react';
import { DotProgress } from '@digital-ai/dot-components';

import { mountWithStoreAndTheme, ReactWrapper, preparePortalContainer } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { AnalyticsDashboardPage, AnalyticsDashboardPageProps } from './analytics-dashboard-page.component';
import { navigation } from '../../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { analytics } from '../../ducks/analytics.reducer';
import { AnalyticsDashboardPagePortal } from './analytics-dashboard-page.portal';
import { MicrostrategyDashboard } from './microstrategy/microstrategy-dashboard.component';

const { setMenuOpen } = navigation.actions;

const { loadDashboard, clearSelectedDashboard, favoriteDashboardFromDetails } = analytics.actions;

describe('AnalyticsDashboardPage', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const dashboardId = '1';
    const defaultState = {
        analytics: {
            selectedDashboard: undefined,
            intelligenceConfiguration: undefined,
            updatingDashboardIds: [],
        },
        navigation: {
            menuIsOpen: true,
        },
        profile: {
            favoriteDashboards: [],
        },
    };

    const prepareBody = () => {
        const portal = global.document.createElement('div');
        portal.setAttribute('id', 'dashboard-placeholder');
        const body = global.document.querySelector('body');
        body?.appendChild(portal);
    };

    const defaultProps: AnalyticsDashboardPageProps = {
        dashboardId,
        folder: {} as never,
    };

    const mount = (props = defaultProps, state = defaultState) => {
        wrapper = mountWithStoreAndTheme(<AnalyticsDashboardPage {...props} />, dispatch, state);
    };

    const getDotProgress = () => wrapper.find(DotProgress);
    const getPortal = () => wrapper.find(AnalyticsDashboardPagePortal);

    beforeEach(() => {
        preparePortalContainer();
        prepareBody();
    });

    afterEach(() => {
        jest.clearAllMocks();
    });

    it('should dispatch actions on init', () => {
        mount();
        expect(dispatch).toHaveBeenCalledWith(setMenuOpen(false));
        expect(dispatch).toHaveBeenCalledWith(loadDashboard(dashboardId));
    });

    it('should reopen the sidebar if it was open', () => {
        mount();
        dispatch.mockReset();
        wrapper.unmount();
        expect(dispatch).toHaveBeenCalledWith(setMenuOpen(true));
        expect(dispatch).toHaveBeenCalledWith(clearSelectedDashboard());
    });

    it('should not reopen the sidebar if it was closed', () => {
        mount(undefined, { ...defaultState, navigation: { menuIsOpen: false } });
        dispatch.mockReset();
        wrapper.unmount();
        expect(dispatch).not.toHaveBeenCalledWith(setMenuOpen(true));
        expect(dispatch).toHaveBeenCalledWith(clearSelectedDashboard());
    });

    it('should show loader if there is no dashboard loaded', () => {
        mount();
        const progress = getDotProgress();
        expect(progress).toExist();
        expect(progress.props().ariaLabel).toBe('Loading dashboard data');
        expect(progress.props().size).toBe(100);
    });

    it('should include portal', () => {
        mount();
        const portal = getPortal();
        expect(portal).toExist();
        expect(portal.props().dashboardId).toBe(dashboardId);
        expect(portal.props().folderId).toBeUndefined();
    });

    it('should include portal with folder', () => {
        mount({ ...defaultProps, folder: { id: 'Folder123' } as never });
        const portal = getPortal();
        expect(portal).toExist();
        const portalProps = portal.props();
        expect(portalProps.dashboardId).toBe(dashboardId);
        expect(portalProps.folderId).toBe('Folder123');
        expect(portalProps.isUpdating).toBe(false);
    });

    it('should set isUpdating flag to true when dashboard ID is contained in updatingDashboardIds array', () => {
        mount(defaultProps, {
            ...defaultState,
            analytics: {
                ...defaultState.analytics,
                intelligenceConfiguration: { id: 'test' },
                updatingDashboardIds: [dashboardId],
            } as never,
        });
        expect(getPortal().props().isUpdating).toBe(true);
    });

    it('should execute correct action when onFavoriteClick has been invoked', () => {
        mount();
        getPortal().invoke('onFavoriteClick')();
        expect(dispatch).toHaveBeenLastCalledWith(
            favoriteDashboardFromDetails({
                id: dashboardId,
                isUnfavorite: false,
            }),
        );
    });

    it('should render MicrostrategyDashboard if we have a dashboard selected', () => {
        const folder = { id: 'Folder1', title: 'My folder' };
        mount({ ...defaultProps, folder }, {
            ...defaultState,
            analytics: {
                selectedDashboard: { embedConfig: { url: 'url' }, serverUrl: 'http' },
                updatingDashboardIds: [],
            },
        } as never);
        const dashboard = wrapper.find(MicrostrategyDashboard);
        expect(dashboard).toExist();
        expect(dashboard.prop('embedConfig')).toStrictEqual({ url: 'url' });
        expect(dashboard.prop('folder')).toStrictEqual(folder);
        expect(dashboard.prop('serverUrl')).toStrictEqual('http');
    });

    it('should not render MicrostrategyDashboard if we have a dashboard selected', () => {
        mount(defaultProps, {
            ...defaultState,
            analytics: {
                selectedDashboard: undefined,
                updatingDashboardIds: [],
            },
        } as never);
        const dashboard = wrapper.find(MicrostrategyDashboard);
        expect(dashboard).not.toExist();
    });
});
