package com.xebialabs.xlrelease.analytics.api

import com.xebialabs.deployit.checks.Checks.{checkArgument, checkNotNull}
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.analytics.api.PlannerApiImpl.TOO_MANY_ERROR
import com.xebialabs.xlrelease.analytics.service.PlannerService
import com.xebialabs.xlrelease.api.ApiService._
import com.xebialabs.xlrelease.api.v1.PlannerApi
import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.domain.analytics.ProjectedRelease
import com.xebialabs.xlrelease.security.{PermissionChecker, XLReleasePermissions}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.lang.{Long => JLong}
import java.util.{Date, List => JList}
import scala.jdk.CollectionConverters._

object PlannerApiImpl {
  val TOO_MANY_ERROR = s"Number of results per page cannot be more than ${DEFAULT_RESULTS_PER_PAGE}"
}

@Controller
class PlannerApiImpl @Autowired()(plannerService: PlannerService,
                                  permissions: PermissionChecker) extends PlannerApi {

  override def getActiveReleases(page: JLong, resultsPerPage: JLong, kind: String): JList[ProjectedRelease] = {
    checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, TOO_MANY_ERROR)
    permissions.checkAny(ADMIN, XLReleasePermissions.AUDIT_ALL)
    plannerService.planActiveReleases(page, resultsPerPage, ReleaseKind.fromString(kind)).asJava
  }

  override def getCompletedReleases(page: JLong, resultsPerPage: JLong, sinceMillis: JLong, kind: String): JList[ProjectedRelease] = {
    checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, TOO_MANY_ERROR)
    permissions.checkAny(ADMIN, XLReleasePermissions.AUDIT_ALL)
    plannerService.planCompletedReleases(page, resultsPerPage, new Date(sinceMillis), ReleaseKind.fromString(kind)).asJava
  }

  override def getReleasesByIds(releaseIds: JList[String]): JList[ProjectedRelease] = {
    checkNotNull(releaseIds, "releaseIds")
    checkArgument(releaseIds.size() <= DEFAULT_RESULTS_PER_PAGE, TOO_MANY_ERROR)
    permissions.checkAny(ADMIN, XLReleasePermissions.AUDIT_ALL)
    plannerService.planReleases(releaseIds.asScala.toSeq).asJava
  }
}
