import { CardSearch } from '../ducks/analytics.reducer';
import { AnalyticsCardData, NativeFilterConfiguration, PlatformListDashboardResponse } from '../types';

export const calculateNumberOfFiltersApplied = (cardSearch: CardSearch | undefined) => {
    let numberOfFilters = 0;
    const { categories, authorNames } = cardSearch || {};
    if (categories && categories.length > 0) numberOfFilters += categories.length;
    if (authorNames && authorNames.length > 0) numberOfFilters += authorNames.length;
    return numberOfFilters;
};

export const removeCategoryFromFilters = (categoryName: string, filteredCategories: string[] | undefined) =>
    filteredCategories?.filter((currentCategory) => currentCategory !== categoryName);

export const removeAuthorFromFilters = (authorName: string, filteredAuthorNames: string[] | undefined) =>
    filteredAuthorNames?.filter((currentAuthorName) => currentAuthorName !== authorName);

export const convertArrayToString = (input: string[] | undefined): string | undefined => {
    if (input && input.length > 0) {
        return input.join(',');
    } else {
        return undefined;
    }
};

export const isFavouriteCard = (favouriteDashboards: string[], id: string) => favouriteDashboards.includes(id);

export const calculateTitleAndSubtitleForEmptyState = (isFavouriteTab: boolean, isAnalyticsConfigured: boolean): { subtitle: string; title: string } => {
    if (isFavouriteTab) {
        return {
            subtitle: 'To easily add dashboards to your favourites, simply click the star icon located next to each of the dashboards. 🌟',
            title: 'No favourites have been added so far',
        };
    } else {
        return isAnalyticsConfigured
            ? {
                  subtitle: 'Create and manage Analytics dashboards',
                  title: 'Analytics dashboards',
              }
            : {
                  subtitle: 'There are no available dashboards yet.',
                  title: 'No dashboards yet',
              };
    }
};

export const filterCardsBySearch = (cards: Array<AnalyticsCardData>, cardSearch?: CardSearch): Array<AnalyticsCardData> => {
    let filteredCards = cards;
    if (cardSearch) {
        const { name, authorNames, categories } = cardSearch;

        if (name) {
            filteredCards = filteredCards.filter((card) => card.title.toLowerCase().includes(name.toLowerCase()));
        }
        if (authorNames && authorNames.length > 0) {
            filteredCards = filteredCards.filter((card) => authorNames.includes(card.subtitle));
        }
        if (categories && categories.length > 0) {
            filteredCards = filteredCards.filter((card) => card.tags.some((tag) => categories.some((category) => tag.includes(category))));
        }
    }
    return filteredCards;
};

export const hasFolderFilter = (dashboard: PlatformListDashboardResponse, folderColumn: string): boolean => {
    const nativeFilters: Array<NativeFilterConfiguration> = dashboard.filter_configuration ?? [];
    const nativeFilterConfigurationTargets = nativeFilters?.flatMap((filter) => filter.targets) ?? [];
    return nativeFilterConfigurationTargets.filter((target) => target !== undefined).some((target) => target.column?.name === folderColumn);
};

export const mapSupersetToAnalyticsCard = (list: Array<PlatformListDashboardResponse>, folderColumn: string): Array<AnalyticsCardData> => {
    return list.map((item) => ({
        embeddingId: item.external_embedding_id,
        embeddingUrl: item.server_url,
        id: item.id,
        hasFolderFilter: hasFolderFilter(item, folderColumn),
        externalId: parseInt(item.external_id),
        tags: ['TODO'],
        title: item.name,
        subtitle: 'TODO',
        imageSrc: 'TODO',
        contentText: item.description,
    }));
};

export const collectAllAuthors = (_list: Array<PlatformListDashboardResponse>): Array<string> => {
    const allAuthors = new Set<string>();
    // TODO we don't have authors coming from platform
    // list.forEach((item) => {
    //     allAuthors.add(item.changed_by_name);
    // });
    return Array.from(allAuthors);
};

export const getFavouriteIcon = (isFavourite: boolean) => (isFavourite ? 'star-favorites-active' : 'star-favorites-default');
export const getFavouriteText = (isFavourite: boolean) => (isFavourite ? 'Remove from favourites' : 'Add to favourites');
