import React, { useEffect, useState } from 'react';
import { CssCell, CssGrid, ButtonProps, DotIcon, DotEmptyFilterState, DotEmptyState } from '@digital-ai/dot-components';
import { AnalyticsCard } from './analytics-card.component';
import { AnalyticsCardData } from '../types';
import { calculateCellProps } from '../../../../../../../../../core/xlr-ui/app/features/workflow/helper';
import { calculateTitleAndSubtitleForEmptyState, isFavouriteCard } from '../helpers';
import './analytics-cards.component.less';

const analyticsDefaultEmptyStateImg = 'static/@project.version@/styles/img/astronaut_rafiki.svg';
const analyticsConfiguredEmptyStateImg = 'static/@project.version@/styles/img/data_analysis.svg';

export interface AnalyticsCardsProps {
    cards: AnalyticsCardData[];
    folderId?: string;
    isAnalyticsConfigured: boolean;
    isFavouritePage: boolean;
    isSearchOrFilterApplied: boolean;
    profileFavouriteDashboards: string[];
}

export const AnalyticsCards = ({
    cards,
    folderId,
    isAnalyticsConfigured,
    isFavouritePage,
    isSearchOrFilterApplied,
    profileFavouriteDashboards,
}: AnalyticsCardsProps) => {
    const [currentAnalyticsCards, setCurrentAnalyticsCards] = useState<AnalyticsCardData[]>([]);

    useEffect(() => {
        setCurrentAnalyticsCards(cards);
    }, [cards]);

    const handleContextMenuItemClick = (_key: string) => {
        // TODO
    };

    const handleFavoriteClick = (_id: string) => () => {
        // TODO
    };

    const handleOpenDashboardClick = (id: string) => () => {
        if (isAnalyticsConfigured) {
            const analyticsUrl = `analytics/${id}`;
            if (folderId) {
                window.location.href = `#/folders/${folderId}/${analyticsUrl}`;
            } else {
                window.location.href = `#/${analyticsUrl}`;
            }
        } else {
            // pendo guide will be open
        }
    };

    const renderEmptyState = () => {
        const emptyStateClassName = 'analytics-empty-state';
        const emptyStateTitleAndSubtitle = calculateTitleAndSubtitleForEmptyState(isFavouritePage, isAnalyticsConfigured);
        const emptyStateButtonProps: ButtonProps | undefined =
            isAnalyticsConfigured && !isFavouritePage
                ? {
                      children: 'Manage dashboards',
                      endIcon: <DotIcon iconId="open-new-tab" />,
                  }
                : undefined;
        const emptyStateImage = isAnalyticsConfigured && !isFavouritePage ? analyticsConfiguredEmptyStateImg : analyticsDefaultEmptyStateImg;

        return isSearchOrFilterApplied ? (
            <DotEmptyFilterState className={emptyStateClassName} />
        ) : (
            <DotEmptyState
                buttonProps={emptyStateButtonProps}
                className={emptyStateClassName}
                imageAltText="analytics dashboards empty state image"
                imageSrc={emptyStateImage}
                subtitle={emptyStateTitleAndSubtitle.subtitle}
                title={emptyStateTitleAndSubtitle.title}
            />
        );
    };

    const renderCards = () => {
        return currentAnalyticsCards.length > 0 ? (
            <CssGrid className="css-grid">
                {currentAnalyticsCards.map((currentAnalyticsCard, index) => {
                    return (
                        <CssCell {...calculateCellProps(index)} key={currentAnalyticsCard.id}>
                            <AnalyticsCard
                                {...currentAnalyticsCard}
                                isAnalyticsConfigured={isAnalyticsConfigured}
                                isFavorite={isFavouriteCard(profileFavouriteDashboards, currentAnalyticsCard.id)}
                                onContextMenuItemClick={handleContextMenuItemClick}
                                onFavoriteClick={handleFavoriteClick(currentAnalyticsCard.id)}
                                onOpenDashboardClick={handleOpenDashboardClick(currentAnalyticsCard.id)}
                            />
                        </CssCell>
                    );
                })}
            </CssGrid>
        ) : (
            renderEmptyState()
        );
    };

    return <div className="analytics-cards">{renderCards()}</div>;
};
