import React, { useEffect, useState } from 'react';
import { CssCell, CssGrid, DotEmptyState } from '@digital-ai/dot-components';
import { AnalyticsCard } from './analytics-card.component';
import { AnalyticsCardData } from '../types';
import { calculateCellProps } from '../../../../../../../../../core/xlr-ui/app/features/workflow/helper';
import { calculateTitleAndSubtitleForEmptyState, isFavouriteCard } from '../helpers';
import { useAppDispatch } from '../../../../../../../../../core/xlr-ui/app/js/hooks';

import { profile } from '../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.reducer';
import './analytics-cards.component.less';

const { toggleFavouriteDashboard } = profile.actions;

const analyticsEmptyStateImg = 'static/@project.version@/styles/img/astronaut_rafiki.svg';

export interface AnalyticsCardsProps {
    cards: AnalyticsCardData[];
    folderId?: string;
    isAnalyticsConfigured: boolean;
    isFavouritePage: boolean;
    isSearchOrFilterApplied: boolean;
    profileFavouriteDashboards: string[];
}

export const AnalyticsCards = ({
    cards,
    folderId,
    isAnalyticsConfigured,
    isFavouritePage,
    isSearchOrFilterApplied,
    profileFavouriteDashboards,
}: AnalyticsCardsProps) => {
    const [currentAnalyticsCards, setCurrentAnalyticsCards] = useState<AnalyticsCardData[]>([]);

    const dispatch = useAppDispatch();

    const emptyStateTitleAndSubtitle = calculateTitleAndSubtitleForEmptyState(isFavouritePage, isSearchOrFilterApplied);

    useEffect(() => {
        setCurrentAnalyticsCards(cards);
    }, [cards]);

    const handleContextMenuItemClick = (_key: string) => {
        // TODO
    };

    const handleFavoriteClick = (id: string) => () => {
        dispatch(toggleFavouriteDashboard(id));
    };

    const handleOpenDashboardClick = (id: string) => () => {
        if (isAnalyticsConfigured) {
            const analyticsUrl = `analytics/${id}`;
            if (folderId) {
                window.location.href = `#/folders/${folderId}/${analyticsUrl}`;
            } else {
                window.location.href = `#/${analyticsUrl}`;
            }
        } else {
            // pendo guide will be open
        }
    };

    const renderCards = () => {
        return currentAnalyticsCards.length > 0 ? (
            <CssGrid className="css-grid">
                {currentAnalyticsCards.map((currentAnalyticsCard, index) => {
                    return (
                        <CssCell {...calculateCellProps(index)} key={currentAnalyticsCard.id}>
                            <AnalyticsCard
                                {...currentAnalyticsCard}
                                isAnalyticsConfigured={isAnalyticsConfigured}
                                isFavorite={isFavouriteCard(profileFavouriteDashboards, currentAnalyticsCard.id)}
                                onContextMenuItemClick={handleContextMenuItemClick}
                                onFavoriteClick={handleFavoriteClick(currentAnalyticsCard.id)}
                                onOpenDashboardClick={handleOpenDashboardClick(currentAnalyticsCard.id)}
                            />
                        </CssCell>
                    );
                })}
            </CssGrid>
        ) : (
            <DotEmptyState
                className="analytics-empty-state"
                imageAltText="analytics dashboards empty state image"
                imageSrc={analyticsEmptyStateImg}
                subtitle={emptyStateTitleAndSubtitle.subtitle}
                title={emptyStateTitleAndSubtitle.title}
            />
        );
    };

    return <div className="analytics-cards">{renderCards()}</div>;
};
