import { all, call, put, takeLatest, debounce, select } from 'redux-saga/effects';
import { IHttpResponse } from 'angular';
import { analytics, AnalyticsState, CardSearch, getAnalyticsState } from './analytics.reducer';
import { PayloadAction } from '@reduxjs/toolkit';
import { ANALYTICS_AUTHORS, ANALYTICS_CATEGORIES, ANALYTICS_SAMPLE_CARDS, INTELLIGENCE_URL } from '../constants';
import { ClientSettings, FilterQueryParams } from '../../../../../../../../../core/xlr-ui/app/types';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { collectAllAuthors, convertArrayToString, mapSupersetToAnalyticsCard } from '../helpers';
import { ConfigurationService } from '../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';
import { SearchConfigurationResponse } from '../../../../../../../../../core/xlr-ui/app/features/configuration/types';
import { httpGET } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';

const {
    init,
    filterCards,
    setCards,
    setCardSearch,
    setAuthors,
    setCategories,
    setIsAnalyticsConfigured,
    setIntelligenceConfiguration,
    setIsLoading,
    loadDashboard,
    setSelectedDashboard,
} = analytics.actions;

interface AnalyticsClientSettings extends ClientSettings {
    setAnalyticsListFilters: (filter: CardSearch | undefined) => void;
}

export interface DashboardListResponse {
    entity: Array<SupersetListDashboardResponse>;
}

export interface DashboardDetailsResponse {
    entity: SupersetDetailsDashboardResponse;
}

export interface SupersetDetailsDashboardResponse {
    dashboard_id: number;
    uuid: string;
}

export interface SupersetListDashboardResponse {
    changed_by_name: string;
    dashboard_title: string;
    id: number;
    slug: string | null;
    thumbnail_url: string;
    url: string;
}

export function* initSaga(action: PayloadAction<Partial<CardSearch> | undefined>) {
    yield put(setCards([]));
    yield call(checkIfIntelligenceIsConfigured);
    yield put(setCategories(ANALYTICS_CATEGORIES));
    yield call(loadCards);
    const cardSearch = action.payload;
    if (cardSearch) {
        yield put(filterCards(cardSearch));
    }
}

export function* checkIfIntelligenceIsConfigured() {
    const configurationService: ConfigurationService = yield call(getAngularService, 'ConfigurationService');
    const intelligenceServerResponse: SearchConfigurationResponse = yield call(
        [configurationService, configurationService.searchConfiguration],
        'dai.IntelligenceServer',
        INTELLIGENCE_URL,
    );
    const intelligenceServer = intelligenceServerResponse['dai.IntelligenceServer'];
    if (intelligenceServer && intelligenceServer.length) {
        yield put(setIntelligenceConfiguration(intelligenceServer[0]));
    } else {
        yield put(setIsAnalyticsConfigured(false));
    }
}

export function* loadCards() {
    const { intelligenceConfiguration, isAnalyticsConfigured }: AnalyticsState = yield select(getAnalyticsState);
    let cards;
    if (isAnalyticsConfigured) {
        yield put(setIsLoading(true));
        const {
            data: { entity },
        }: IHttpResponse<DashboardListResponse> = yield call(httpGET, `api/extension/intelligence/dashboards?serverId=${intelligenceConfiguration?.id}`);
        cards = mapSupersetToAnalyticsCard(entity);
        yield put(setAuthors(collectAllAuthors(entity)));
    } else {
        yield put(setAuthors(ANALYTICS_AUTHORS));
        cards = ANALYTICS_SAMPLE_CARDS;
    }
    yield put(setIsLoading(false));
    yield put(setCards(cards));
}

export function* filterCardsSaga(action: PayloadAction<CardSearch>) {
    const cardSearch = action.payload;
    yield put(setCardSearch(cardSearch));
}

export function* setCardSearchSaga(action: PayloadAction<Partial<CardSearch> | undefined>) {
    const FiltersQueryParams: FilterQueryParams = yield call(getAngularService, 'FiltersQueryParams');
    const clientSettings: AnalyticsClientSettings = yield call(getAngularService, 'ClientSettings');
    const cardSearch = action.payload;
    clientSettings.setAnalyticsListFilters(cardSearch);
    FiltersQueryParams.update({
        ...cardSearch,
        categories: convertArrayToString(cardSearch?.categories),
        authorNames: convertArrayToString(cardSearch?.authorNames),
    });
}

export function* loadDashboardSaga(action: PayloadAction<number>) {
    yield call(checkIfIntelligenceIsConfigured);
    const { intelligenceConfiguration, isAnalyticsConfigured }: AnalyticsState = yield select(getAnalyticsState);
    const dashboardId = action.payload;
    if (isAnalyticsConfigured) {
        const {
            data: { entity },
        }: IHttpResponse<DashboardDetailsResponse> = yield call(
            httpGET,
            `api/extension/intelligence/dashboard?serverId=${intelligenceConfiguration?.id}&dashboardId=${dashboardId}`,
        );
        yield put(setSelectedDashboard(entity));
    }
}

export function* analyticsSaga() {
    yield all([
        takeLatest(init, initSaga),
        debounce(100, filterCards, filterCardsSaga),
        takeLatest(setCardSearch, setCardSearchSaga),
        takeLatest(loadDashboard, loadDashboardSaga),
    ]);
}
