import React from 'react';
import { ReactWrapper, mountWithStoreAndTheme, mockResizeObserver } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { AnalyticsCards, AnalyticsCardsProps } from './analytics-cards.component';
import { ANALYTICS_SAMPLE_CARDS } from '../constants';
import { CssCell, CssGrid, DotEmptyState } from '@digital-ai/dot-components';
import { AnalyticsCard, AnalyticsCardProps } from './analytics-card.component';
import { profile } from '../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.reducer';

const { toggleFavouriteDashboard } = profile.actions;

describe('AnalyticsCards', () => {
    let wrapper: ReactWrapper;

    const defaultProps: AnalyticsCardsProps = {
        cards: ANALYTICS_SAMPLE_CARDS,
        isFavouritePage: false,
        isSearchOrFilterApplied: false,
        isAnalyticsConfigured: false,
        profileFavouriteDashboards: ['1'],
    };

    const dispatch = jest.fn();

    const getAnalyticsCardsWrapper = () => wrapper.find('div.analytics-cards');
    const getCssGrid = () => wrapper.find(CssGrid);
    const getCssCells = () => wrapper.find(CssCell);
    const getAnalyticsCards = () => wrapper.find(AnalyticsCard);
    const getEmptyState = () => wrapper.find(DotEmptyState);

    const mount = (props = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<AnalyticsCards {...props} />, dispatch);
    };

    beforeEach(() => {
        mockResizeObserver();
    });

    it('should render with correct elements', () => {
        mount();
        expect(getAnalyticsCardsWrapper()).toExist();
        expect(getCssGrid()).toExist();

        const cssCells = getCssCells();
        expect(cssCells.length).toBe(ANALYTICS_SAMPLE_CARDS.length);

        cssCells.forEach((cssCell: ReactWrapper<typeof CssCell>) => {
            expect(cssCell).toExist();
        });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.length).toBe(ANALYTICS_SAMPLE_CARDS.length);

        analyticsCards.forEach((cssCell: ReactWrapper<AnalyticsCardProps>) => {
            expect(cssCell).toExist();
        });
    });

    it('should handle favourite toggle', () => {
        mount({ ...defaultProps, isFavouritePage: true });
        const analyticCard = getAnalyticsCards().at(0);
        analyticCard.invoke('onFavoriteClick')();
        expect(dispatch).toHaveBeenLastCalledWith(toggleFavouriteDashboard(analyticCard.props().id.toString()));
    });

    it('should render empty state with correct title when on "Favourites" tab and no data is present', () => {
        mount({ ...defaultProps, cards: [], isFavouritePage: true, profileFavouriteDashboards: [] });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.length).toBe(0);

        const emptyState = getEmptyState();
        expect(emptyState).toExist();
        expect(emptyState.props().title).toBe('No favourites have been added so far');
    });

    it('should render empty state with correct title when on "All dashboards" tab and no data is present', () => {
        mount({ ...defaultProps, cards: [], profileFavouriteDashboards: [] });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.length).toBe(0);

        const emptyState = getEmptyState();
        expect(emptyState).toExist();
        expect(emptyState.props().title).toBe('No dashboards yet');
    });

    it('should render empty state with correct title when search is applied and no data is present', () => {
        mount({ ...defaultProps, cards: [], isSearchOrFilterApplied: true, profileFavouriteDashboards: [] });

        const analyticsCards = getAnalyticsCards();
        expect(analyticsCards.length).toBe(0);

        const emptyState = getEmptyState();
        expect(emptyState).toExist();
        expect(emptyState.props().title).toBe('No results found');
    });

    it('should handle open click event when isAnalyticsConfigured is true', () => {
        mount({ ...defaultProps, isAnalyticsConfigured: true });
        const analyticsCard = getAnalyticsCards().at(0);
        analyticsCard.invoke('onOpenDashboardClick')();
        expect(window.location.href).toBe('http://localhost/#/analytics/1');
    });
});
