import React, { useEffect, useLayoutEffect } from 'react';
import { Provider } from 'react-redux';
import { embedDashboard } from '@superset-ui/embedded-sdk';
import { DotProgress, DotThemeProvider } from '@digital-ai/dot-components';
import { store } from '../../../analytics-store';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getNavigationState } from '../../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.selectors';
import { analytics, getAnalyticsState } from '../../ducks/analytics.reducer';
import { httpGET } from '../../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { INTELLIGENCE_URL } from '../../constants';
import { AnalyticsDashboardPagePortal } from './analytics-dashboard-page.portal';
import { useCloseSidebar } from '../../../../../../../../../../core/xlr-ui/app/features/common/hooks';
import './analytics-dashboard-page.component.less';

const { loadDashboard, clearSelectedDashboard } = analytics.actions;

export const dashboardDivId = 'dashboard-placeholder';

export interface AnalyticsDashboardPageProps {
    dashboardId: number;
}

export const AnalyticsDashboardPageProvider = (props: AnalyticsDashboardPageProps) => {
    return (
        <Provider store={store}>
            <DotThemeProvider>
                <AnalyticsDashboardPage {...props} />
            </DotThemeProvider>
        </Provider>
    );
};
export const AnalyticsDashboardPage = ({ dashboardId }: AnalyticsDashboardPageProps) => {
    const { menuIsOpen } = useAppSelector(getNavigationState);
    const { selectedDashboard, intelligenceConfiguration } = useAppSelector(getAnalyticsState);
    const dispatch = useAppDispatch();

    useCloseSidebar(menuIsOpen, dispatch);
    useEffect(() => {
        dispatch(loadDashboard(dashboardId));
        return () => {
            dispatch(clearSelectedDashboard());
        };
    }, []);

    useLayoutEffect(() => {
        if (selectedDashboard) {
            (async () => {
                const superSetTileElement = document.getElementById(dashboardDivId);
                if (superSetTileElement) {
                    await embedDashboard({
                        id: selectedDashboard.uuid, // given by the Superset embedding UI
                        supersetDomain: INTELLIGENCE_URL,
                        mountPoint: superSetTileElement, // html element in which iframe render
                        fetchGuestToken: getToken,
                        dashboardUiConfig: {
                            hideTitle: false,
                            hideChartControls: false,
                            hideTab: false,
                        },
                    });
                }
            })();
        }
    }, [selectedDashboard]);

    const getToken = () => {
        const url = `api/extension/intelligence/token?serverId=${intelligenceConfiguration?.id}&dashboardUuid=${selectedDashboard?.uuid}`;
        return httpGET(url).then((response) => response.data.entity);
    };

    const renderLoader = () => {
        return (
            <div className="dashboard-loader">
                <DotProgress ariaLabel="Loading dashboard data" size={100} />
            </div>
        );
    };

    return (
        <div className="analytics-dashboard-page">
            <AnalyticsDashboardPagePortal dashboardId={dashboardId} />
            {selectedDashboard && <div id={dashboardDivId}></div>}
            {!selectedDashboard && renderLoader()}
        </div>
    );
};
