import React, { MouseEvent, useEffect, useState } from 'react';
import { Provider } from 'react-redux';
import {
    DotAvatar,
    DotBadge,
    DotIcon,
    DotIconButton,
    DotInputText,
    DotTabs,
    DotThemeProvider,
    DotTypography,
    DotChip,
    DotButton,
    DotSkeleton,
} from '@digital-ai/dot-components';
import { store } from '../../analytics-store';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getLicense } from '../../../../../../../../../core/xlr-ui/app/features/license/ducks/license-warning.selectors';
import { getProfile } from '../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.selectors';
import { Folder, License, UserProfileSettings } from '../../../../../../../../../core/xlr-ui/app/types';
import { AnalyticsCards } from './analytics-cards.component';
import { analytics, CardSearch, getAnalyticsState } from '../ducks/analytics.reducer';
import { calculateNumberOfFiltersApplied, filterCardsBySearch, isFavouriteCard, removeAuthorFromFilters, removeCategoryFromFilters } from '../helpers';
import { AnalyticsPageFilterDrawer } from './analytics-page-filter-drawer.component';
import { AnalyticsCardsSkeleton } from './analytics-cards.skeleton';
import { AnalyticsPagePortal } from './analytics-page.portal';
import './analytics-page.component.less';

interface AnalyticsPageProps {
    filters: CardSearch;
    folder?: Folder;
}

export const AnalyticsPageProvider = (props: AnalyticsPageProps) => {
    return (
        <Provider store={store}>
            <DotThemeProvider>
                <AnalyticsPage {...props} />
            </DotThemeProvider>
        </Provider>
    );
};

const ANALYTICS_PAGE_TABS = [{ label: 'All dashboards' }, { label: 'Favourites' }];

const { init, filterCards, setIsDrawerOpened } = analytics.actions;

const LICENSE_EDITIONS_FOR_BANNER = ['Essentials', 'Pro', 'Premium', 'Trial'];

export const AnalyticsPage = ({ filters, folder }: AnalyticsPageProps) => {
    const dispatch = useAppDispatch();
    const [selectedTab, setSelectedTab] = useState(0);
    const profileSettings: UserProfileSettings = useAppSelector(getProfile);
    const license: License | undefined = useAppSelector(getLicense);
    const { authors, isDrawerOpened, cards, categories, cardSearch, isAnalyticsConfigured, isLoading } = useAppSelector(getAnalyticsState);
    const isBannerVisible = LICENSE_EDITIONS_FOR_BANNER.includes(license?.edition || 'Community') && !isAnalyticsConfigured;
    const numberOfFiltersApplied = calculateNumberOfFiltersApplied(cardSearch);
    const { categories: filteredCategories, authorNames: filteredAuthors } = cardSearch || {};
    const isAnyCategoryFilterPresent = !!(filteredCategories && filteredCategories.length > 0);
    const isAnyAuthorFilterPresent = !!(filteredAuthors && filteredAuthors.length > 0);
    const isAnyFilterPresent = isAnyAuthorFilterPresent || isAnyCategoryFilterPresent;
    const isSearchOrFilterApplied = (cardSearch && cardSearch.name !== '') || isAnyFilterPresent;

    useEffect(() => {
        dispatch(init(filters));
    }, []);

    const getCardSearch = (currentCardSearch: CardSearch) => (cardSearch ? { ...cardSearch, ...currentCardSearch } : currentCardSearch);

    const handleDrawerOpen = (event: MouseEvent<HTMLButtonElement>) => {
        if (isDrawerOpened) return;
        event.stopPropagation();
        dispatch(setIsDrawerOpened(true));
    };
    const handleDrawerClose = () => dispatch(setIsDrawerOpened(false));
    const handleCardFilterChange = (currentCardSearch: CardSearch) => dispatch(filterCards(currentCardSearch));
    const handleTabChange = (value: number) => setSelectedTab(value);
    const handleClearAllFilters = () =>
        dispatch(
            filterCards({
                ...cardSearch,
                categories: undefined,
                authorNames: undefined,
            }),
        );

    const filteredCards = filterCardsBySearch(cards, cardSearch);
    const filteredFavouritesCards = filteredCards.filter((card) => isFavouriteCard(profileSettings.favouriteDashboards, card.id));

    const getAnalyticsTabsWithNumbers = () => {
        const allDashboardsCount = filteredCards.length;
        return ANALYTICS_PAGE_TABS.map((tab, index) => {
            const tabCount = index === 0 ? allDashboardsCount : filteredFavouritesCards.length;
            return { label: `${tab.label} (${tabCount})` };
        });
    };

    const renderLoadingPage = () => {
        return (
            <div className="analytics-page">
                <AnalyticsPagePortal isAnalyticsConfigured={isAnalyticsConfigured} />
                <div className="analytics-page-content">
                    <div className="content-header">
                        <DotTypography data-testid="page-title" variant="h1">
                            Digital.ai Analytics
                        </DotTypography>
                        <div className="content-header-filter">
                            <DotSkeleton height="40px" variant="rectangular" width="200px" />
                        </div>
                    </div>

                    <DotTabs className="analytics-tabs" initialValue={selectedTab} onChange={handleTabChange} tabs={ANALYTICS_PAGE_TABS} />
                    <div>
                        {selectedTab === 0 && <AnalyticsCardsSkeleton />}
                        {selectedTab === 1 && <AnalyticsCardsSkeleton />}
                    </div>
                </div>
            </div>
        );
    };

    const renderPage = () => {
        const hidePortalButtons = isAnalyticsConfigured && selectedTab === 0 && !cards.length && !isSearchOrFilterApplied;

        return (
            <div className="analytics-page">
                {!hidePortalButtons && <AnalyticsPagePortal isAnalyticsConfigured={isAnalyticsConfigured} />}
                <div className="analytics-page-content">
                    {isBannerVisible && (
                        <div className="analytics-banner">
                            <DotAvatar alt="crown icon" className="analytics-banner-avatar" iconId="crown" />
                            <DotTypography variant="body1">
                                The upcoming release will include the availability of new Analytics dashboards in the Premium edition. For more information,
                                please contact your Customer Success Manager.
                            </DotTypography>
                        </div>
                    )}
                    <div className="content-header">
                        <DotTypography data-testid="page-title" variant="h1">
                            Digital.ai Analytics
                        </DotTypography>
                        <div className="content-header-filter">
                            <DotInputText
                                className="name-filter"
                                defaultValue={filters.name}
                                id="name-filter"
                                name="name-filter"
                                onChange={(e) => handleCardFilterChange(getCardSearch({ name: e.target.value }))}
                                onClear={() => handleCardFilterChange(getCardSearch({ name: '' }))}
                                placeholder="Filter by name ..."
                                startIcon={<DotIcon data-testid="name-filter-search-icon" fontSize="small" iconId="search" />}
                            />
                            <DotBadge badgeColor="#d61f21" badgeContent={numberOfFiltersApplied} overlap="circular" variant="standard">
                                <DotIconButton iconId="filter" onClick={handleDrawerOpen} />
                            </DotBadge>
                        </div>
                    </div>

                    {isAnyFilterPresent && (
                        <div className="aligned-flex-with-gap" data-testid="filters-row">
                            {filteredCategories && filteredCategories.length > 0 && (
                                <>
                                    <DotTypography data-testid="category-filters-title" variant="subtitle2">
                                        Categories:
                                    </DotTypography>
                                    {filteredCategories.map((categoryName) => (
                                        <DotChip
                                            data-testid={`category-filters-${categoryName}`}
                                            key={categoryName}
                                            onDelete={() =>
                                                handleCardFilterChange(
                                                    getCardSearch({ categories: removeCategoryFromFilters(categoryName, filteredCategories) }),
                                                )
                                            }
                                            size="small"
                                        >
                                            {categoryName}
                                        </DotChip>
                                    ))}
                                </>
                            )}
                            {filteredAuthors && filteredAuthors.length > 0 && (
                                <>
                                    <DotTypography data-testid="author-filters-title" variant="subtitle2">
                                        Authors:
                                    </DotTypography>
                                    {filteredAuthors.map((authorName) => (
                                        <DotChip
                                            avatar={<DotAvatar alt={`Image for user ${authorName}`} size="small" text={authorName} type="text" />}
                                            data-testid={`author-filters-${authorName}`}
                                            key={authorName}
                                            onDelete={() =>
                                                handleCardFilterChange(getCardSearch({ authorNames: removeAuthorFromFilters(authorName, filteredAuthors) }))
                                            }
                                            size="small"
                                        >
                                            {authorName}
                                        </DotChip>
                                    ))}
                                </>
                            )}
                            <DotButton className="chipy-btn" data-testid="clear-all-btn" onClick={handleClearAllFilters} type="text">
                                Clear all
                            </DotButton>
                        </div>
                    )}
                    <DotTabs className="analytics-tabs" initialValue={selectedTab} onChange={handleTabChange} tabs={getAnalyticsTabsWithNumbers()} />
                    <div className="analytics-tabs-wrapper">
                        {selectedTab === 0 && (
                            <AnalyticsCards
                                cards={filteredCards}
                                folderId={folder?.id}
                                isAnalyticsConfigured={isAnalyticsConfigured}
                                isFavouritePage={false}
                                isSearchOrFilterApplied={isSearchOrFilterApplied}
                                profileFavouriteDashboards={profileSettings.favouriteDashboards}
                            />
                        )}
                        {selectedTab === 1 && (
                            <AnalyticsCards
                                cards={filteredFavouritesCards}
                                folderId={folder?.id}
                                isAnalyticsConfigured={isAnalyticsConfigured}
                                isFavouritePage={true}
                                isSearchOrFilterApplied={isSearchOrFilterApplied}
                                profileFavouriteDashboards={profileSettings.favouriteDashboards}
                            />
                        )}
                    </div>
                </div>
                <AnalyticsPageFilterDrawer
                    authors={authors}
                    cardSearch={cardSearch}
                    categories={categories}
                    isDrawerOpened={isDrawerOpened}
                    numberOfFiltersApplied={numberOfFiltersApplied}
                    onCardFilter={handleCardFilterChange}
                    onClearAll={handleClearAllFilters}
                    onDrawerClose={handleDrawerClose}
                />
            </div>
        );
    };

    return (
        <>
            {isLoading && renderLoadingPage()}
            {!isLoading && renderPage()}
        </>
    );
};
