import { SagaIterator } from 'redux-saga';
import { all, call, put, takeLatest, debounce, select } from 'redux-saga/effects';
import {
    analyticsSaga,
    checkIfIntelligenceIsConfigured,
    filterCardsSaga,
    initSaga,
    loadCards,
    setCardSearchSaga,
    loadDashboardSaga,
    toaster,
} from './analytics.saga';
import { analytics, CardSearch, getAnalyticsState } from './analytics.reducer';
import { ANALYTICS_AUTHORS, ANALYTICS_CATEGORIES, ANALYTICS_SAMPLE_CARDS, INTELLIGENCE_SERVER_TITLE } from '../constants';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { httpGET } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';

const {
    init,
    filterCards,
    setCategories,
    setAuthors,
    setCards,
    setCardSearch,
    setIntelligenceConfiguration,
    setIsAnalyticsConfigured,
    setIsLoading,
    loadDashboard,
    setSelectedDashboard,
} = analytics.actions;

describe('analyticsSaga', () => {
    const cards = [
        {
            contentText: 'Website Traffic Report for October',
            id: 'card1',
            imageSrc: 'test',
            isFavorite: true,
            subtitle: 'Digital.ai',
            tags: [ANALYTICS_CATEGORIES[0], ANALYTICS_CATEGORIES[1], ANALYTICS_CATEGORIES[2]],
            title: 'Onboarding plan and progress A',
        },
        {
            contentText: 'Website Traffic Report for December',
            id: 'card1',
            imageSrc: 'test',
            isFavorite: true,
            subtitle: 'Custom',
            tags: [ANALYTICS_CATEGORIES[1]],
            title: 'Onboarding plan and progress B',
        },
    ];

    it('should yield all effects', () => {
        const gen: SagaIterator = analyticsSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(init, initSaga),
                debounce(100, filterCards, filterCardsSaga),
                takeLatest(setCardSearch, setCardSearchSaga),
                takeLatest(loadDashboard, loadDashboardSaga),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('initSaga', () => {
        const categories = ANALYTICS_CATEGORIES;
        const cardSearch = { name: 'abc' };

        it('should call proper saga', () => {
            const gen: SagaIterator = initSaga({ payload: cardSearch, type: 'test' });
            expect(gen.next().value).toStrictEqual(put(setCards([])));
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(put(setCategories(categories)));
            expect(gen.next().value).toStrictEqual(call(loadCards));
            expect(gen.next(cards).value).toStrictEqual(put(filterCards(cardSearch)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('checkIfIntelligenceIsConfigured', () => {
        const configurationService = {
            searchConfiguration: jest.fn(),
        };

        it('should set analytics configured to true if there is intelligence server', () => {
            const gen: SagaIterator = checkIfIntelligenceIsConfigured();
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ConfigurationService'));
            expect(gen.next(configurationService).value).toStrictEqual(
                call([configurationService, configurationService.searchConfiguration], 'dai.IntelligenceServer', INTELLIGENCE_SERVER_TITLE),
            );
            expect(gen.next({ 'dai.IntelligenceServer': [{ id: 'FakeServer' }] }).value).toStrictEqual(
                put(setIntelligenceConfiguration({ id: 'FakeServer' } as never)),
            );
        });

        it('should set analytics configured to false if there is no intelligence server', () => {
            const gen: SagaIterator = checkIfIntelligenceIsConfigured();
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ConfigurationService'));
            expect(gen.next(configurationService).value).toStrictEqual(
                call([configurationService, configurationService.searchConfiguration], 'dai.IntelligenceServer', INTELLIGENCE_SERVER_TITLE),
            );
            expect(gen.next({ 'dai.IntelligenceServer': [] }).value).toStrictEqual(put(setIsAnalyticsConfigured(false)));
        });
    });

    describe('loadCards', () => {
        const authors = ANALYTICS_AUTHORS;

        it('should load the cards from the api', () => {
            const state = { isAnalyticsConfigured: true, intelligenceConfiguration: { id: 'MyServerId', folderColumnName: 'FOLDER_COLUMN' } };
            const response = {
                data: {
                    entity: [
                        {
                            id: 1,
                            external_id: '1',
                            external_object: {
                                json_metadata: {
                                    native_filter_configuration: [
                                        {
                                            name: '',
                                            id: '',
                                            targets: [{ column: { name: 'FOLDER_COLUMN' } }],
                                        },
                                    ],
                                },
                            },
                        },
                    ],
                },
            };

            const gen: SagaIterator = loadCards();
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state).value).toStrictEqual(put(setIsLoading(true)));
            expect(gen.next().value).toStrictEqual(call(httpGET, `api/extension/intelligence/dashboards?serverId=MyServerId`));
            expect(gen.next(response).value).toStrictEqual(put(setAuthors([])));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(gen.next().value).toStrictEqual(
                put(
                    setCards([
                        {
                            contentText: undefined,
                            embeddingId: undefined,
                            embeddingUrl: undefined,
                            title: undefined,
                            externalId: 1,
                            hasFolderFilter: true,
                            id: 1,
                            imageSrc: 'TODO',
                            subtitle: 'TODO',
                            tags: ['TODO'],
                        },
                    ] as never),
                ),
            );
            expect(gen.next().done).toBe(true);
        });

        it('should load the cards from the mock', () => {
            const state = { isAnalyticsConfigured: false, intelligenceConfiguration: undefined };
            const gen: SagaIterator = loadCards();
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next(state).value).toStrictEqual(put(setAuthors(authors)));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(gen.next().value).toStrictEqual(put(setCards(ANALYTICS_SAMPLE_CARDS)));
            expect(gen.next()).toStrictEqual({ value: ANALYTICS_SAMPLE_CARDS, done: true });
        });
    });

    describe('filterCardsSaga', () => {
        it('should call proper saga', () => {
            const cardSearch = { name: 'progress B' };
            const gen: SagaIterator = filterCardsSaga({ payload: cardSearch, type: 'test' });
            expect(gen.next(cards).value).toStrictEqual(put(setCardSearch(cardSearch)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('setCardSearchSaga', () => {
        const setAnalyticsListFilters = jest.fn();
        const update = jest.fn();
        const FiltersQueryParams = {
            update,
        };
        const ClientSettings = {
            setAnalyticsListFilters,
        };

        const runSaga = (payload: CardSearch) => {
            const gen: SagaIterator = setCardSearchSaga({ payload, type: 'test' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'FiltersQueryParams'));
            expect(gen.next(FiltersQueryParams).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(ClientSettings).done).toBe(true);
        };

        it('should update the query params and local storage when saving the filters', () => {
            runSaga({ name: 'progress B' });
            expect(update).toHaveBeenCalledWith({ authorNames: undefined, categories: undefined, name: 'progress B' });
            expect(setAnalyticsListFilters).toHaveBeenCalledWith({ authorNames: undefined, categories: undefined, name: 'progress B' });
        });

        it('should convert categories and authors as csv', () => {
            runSaga({ name: 'progress B', categories: ['cat1', 'cat2'], authorNames: ['author1'] });
            expect(update).toHaveBeenCalledWith({ authorNames: 'author1', categories: 'cat1,cat2', name: 'progress B' });
            expect(setAnalyticsListFilters).toHaveBeenCalledWith({ authorNames: ['author1'], categories: ['cat1', 'cat2'], name: 'progress B' });
        });
    });

    describe('loadDashboardSaga', () => {
        it('should select the dashboard if found', () => {
            const gen: SagaIterator = loadDashboardSaga({ payload: '1', type: '' });
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(
                gen.next({
                    isAnalyticsConfigured: true,
                    cards: [{ id: '1' }],
                }).value,
            ).toStrictEqual(put(setSelectedDashboard({ id: '1' } as never)));
            expect(gen.next().done).toBe(true);
        });
        it('should show toastr of not found if the dashboard is not found', () => {
            const gen: SagaIterator = loadDashboardSaga({ payload: '1', type: '' });
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(
                gen.next({
                    isAnalyticsConfigured: true,
                    cards: [{ id: '2' }],
                }).value,
            ).toStrictEqual(call(toaster.error, "Dashboard [1] can't be found"));
            expect(gen.next().done).toBe(true);
        });
        it('should reload cards', () => {
            const gen: SagaIterator = loadDashboardSaga({ payload: '1', type: '' });
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(
                gen.next({
                    isAnalyticsConfigured: true,
                    cards: [],
                }).value,
            ).toStrictEqual(call(loadCards));

            expect(gen.next([{ id: '1' }]).value).toStrictEqual(put(setSelectedDashboard({ id: '1' } as never)));
            expect(gen.next().done).toBe(true);
        });

        it('should not not do anything if isAnalyticsConfigured is not configured', () => {
            const gen: SagaIterator = loadDashboardSaga({ payload: '1', type: '' });
            expect(gen.next().value).toStrictEqual(call(checkIfIntelligenceIsConfigured));
            expect(gen.next().value).toStrictEqual(select(getAnalyticsState));
            expect(gen.next({ isAnalyticsConfigured: false }).done).toBe(true);
        });
    });
});
