package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.activity.ActivityOps
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder}
import com.xebialabs.xlrelease.repository.sql.persistence.ActivityLogSchema.ACTIVITY_LOGS
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.USER_PROFILE

import java.util.Date
import scala.collection.mutable


object ActivityLogSqlBuilder {
  val STMT_ACTIVITY_LOG_SELECT: String =
    s"""SELECT
       | ${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.ID},
       | ${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.ACTIVITY_TYPE},
       | ${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.MESSAGE},
       | ${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.USERNAME},
       | ${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.EVENT_TIME},
       | ${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.TARGET_TYPE},
       | ${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.TARGET_ID},
       | ${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.DATA_ID}
       |   FROM ${ACTIVITY_LOGS.TABLE}""".stripMargin

}

class ActivityLogSqlBuilder(implicit dialect: Dialect) extends SqlBuilder[ActivityLogSqlBuilder] {

  import ActivityLogSqlBuilder._

  def select(): ActivityLogSqlBuilder = {
    super.select(STMT_ACTIVITY_LOG_SELECT)
  }

  def withContainer(containerId: CiId): ActivityLogSqlBuilder = {
    if (containerId != null && !containerId.isEmpty) {
      conditions += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.CONTAINER_ID} = ?", Seq(containerId))
    }
    this
  }

  def withFilter(filter: String): ActivityLogSqlBuilder = {
    if (filter != null && filter.nonEmpty) {
      addJoin(s"LEFT JOIN ${USER_PROFILE.TABLE} ON ${USER_PROFILE.TABLE}.${USER_PROFILE.USERNAME} = ${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.USERNAME}")
      val (parameters, orConditions) = likeOrConditions(Seq(
        s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.TARGET_ID}",
        s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.MESSAGE}",
        s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.USERNAME}",
        s"${USER_PROFILE.TABLE}.${USER_PROFILE.FULL_NAME}"),
        filter)
      if (orConditions.nonEmpty) {
        if (filter.toUpperCase() == "SYSTEM") {
          orConditions += s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.USERNAME} IS NULL"
        }
        conditions += Sql(orConditions.mkString(start = "(", sep = " OR ", end = ")"), parameters)
      }

    }
    this
  }

  def withFrom(from: Date): ActivityLogSqlBuilder = {
    if (from != null) {
      conditions += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.EVENT_TIME} >= ?", Seq(from))
    }
    this
  }

  def withTo(to: Date): ActivityLogSqlBuilder = {
    if (to != null) {
      conditions += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.EVENT_TIME} <= ?", Seq(to))
    }
    this
  }

  def withActivityType(activityType: ActivityOps): ActivityLogSqlBuilder = {
    if (activityType != null) {
      withActivityType(activityType.getName)
    }
    this
  }

  def withActivityType(activityType: String): ActivityLogSqlBuilder = {
    if (activityType != null) {
      conditions += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.ACTIVITY_TYPE} = ?", Seq(activityType))
    }
    this
  }

  def withUsernames(usernames: Seq[String]): ActivityLogSqlBuilder = {
    val or = mutable.ListBuffer[Sql]()
    if (usernames.nonEmpty) {
      or += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.USERNAME} IN (${usernames.map(_ => "?").mkString(",")})", usernames)
      if (usernames.contains("SYSTEM")) {
        or += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.USERNAME} IS NULL", Seq())
        or += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.USERNAME} = ?", Seq("SYSTEM"))
      }
    }
    if (or.nonEmpty) {
      conditions += Sql(s"((${or.map(_.sql).mkString(") OR (")}))", or.flatMap(_.parameters))
    }
    this
  }

  def withActivityTypes(activityTypes: Seq[String]): ActivityLogSqlBuilder = {
    if (activityTypes.nonEmpty) {
      conditions += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.ACTIVITY_TYPE} IN (${activityTypes.map(_ => "?").mkString(",")})", activityTypes)
    }
    this
  }

  def withActivityTypes(activityTypes: Seq[ActivityOps], otherTypesExcept: Option[Seq[ActivityOps]]): ActivityLogSqlBuilder = {
    val or = mutable.ListBuffer[Sql]()
    if (activityTypes.nonEmpty) {
      or += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.ACTIVITY_TYPE} IN (${activityTypes.map(_ => "?").mkString(",")})", activityTypes.map(_.getName()))
    } else {
      or += Sql("0 = 1", Seq())
    }
    otherTypesExcept.foreach(ops => {
      or += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.ACTIVITY_TYPE} NOT IN (${ops.map(_ => "?").mkString(",")})", ops.map(_.getName()))
    })
    conditions += Sql(s"((${or.map(_.sql).mkString(") OR (")}))", or.flatMap(_.parameters))
    this
  }

  def withTargetId(targetId: String): ActivityLogSqlBuilder = {
    if (targetId != null && targetId.nonEmpty) {
      conditions += Sql(s"${ACTIVITY_LOGS.TABLE}.${ACTIVITY_LOGS.TARGET_ID} = ?", Seq(targetId))
    }
    this
  }

  override def newInstance: ActivityLogSqlBuilder = new ActivityLogSqlBuilder()

}
