import React, { KeyboardEvent, MouseEvent, useEffect } from 'react';
import { Provider } from 'react-redux';
import { DotSearch, DotSwitch, DotThemeProvider, DotTypography, useKeyPress } from '@digital-ai/dot-components';
import { ActivityLogsFilterDrawer } from './activity-logs-filter-drawer.component';
import { store } from '../../../';
import { ActivityLogsContainerEnum } from '../types';
import { activityLogs as activityLogsSlice, getActivityLogsState } from '../ducks/activity-logs.reducer';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { ActivityLogsTable } from './table/activity-logs-table.component';
import { countFiltersApplied, getActivityLogsCategories } from '../helper';
import { getActivityTypeLabel } from '../../../helper';
import { getProfile } from '@xlr-ui/app/features/profile/ducks/profile.selectors';
import { ActivityLogsFilters } from './filters/activity-logs-filters.component';
import { RegionalSettings } from '@xlr-ui/app/js/locale/regional-settings';
import { calculateDotDateFormat, calculateDotTimeFormat } from '@xlr-ui/app/features/common/helper/date';
import { ActivityLogsEventDialog } from './table/activity-logs-event-dialog.component';
import { ActionToolbarPortalContainer } from '@xlr-ui/app/features/main-navigation/action-toolbar/components/action-toolbar.portal';
import { Release } from '@xlr-ui/app/types';
import { getAllTasks } from '@xlr-ui/app/features/tasks/helper/task.helper';
import { ActivityLogsFilterSettings } from '../../../types';
import { FilterButton } from '@xlr-ui/app/features/common/components/filter-button/filter-button.component';
import { DEFAULT_FILTER_SETTINGS } from '../../../constants';
import './activity-logs.component.less';

const { init, reset, setIsDrawerOpened, loadFilteredAndPagedLogs, setHighlightImportant, closeEventDialog, loadEventDialog } = activityLogsSlice.actions;

export interface ActivityLogsProps {
    containerEnum: ActivityLogsContainerEnum;
    containerId: string;
    filterSettings: ActivityLogsFilterSettings;
    release?: Release;
}

export const ActivityLogs = (props: ActivityLogsProps) => (
    <Provider store={store}>
        <DotThemeProvider>
            <ActivityLogsContent {...props} />
            <ActionToolbarPortalContainer helpLink="concept/release-activity-logs" />
        </DotThemeProvider>
    </Provider>
);

export const ActivityLogsContent = ({ release, filterSettings, containerId, containerEnum }: ActivityLogsProps) => {
    const dispatch = useAppDispatch();
    const {
        activityTypes: activityTypesFromReducer,
        eventDetails,
        isEventDialogOpened,
        isDrawerOpened,
        isLastPage,
        isLoading,
        logs,
        logsFilter,
        page,
        users,
        variables,
    } = useAppSelector(getActivityLogsState);
    const { dateFormat: profileDateFormat, timeFormat: profileTimeFormat } = useAppSelector(getProfile);
    const dateFormat = calculateDotDateFormat(RegionalSettings.getDateFormats(profileDateFormat).mediumDate, 'medium');
    const timeFormat = calculateDotTimeFormat(RegionalSettings.getTimeFormats(profileTimeFormat).mediumTimeWithMs, 'mediumWithMs');
    const filters = logsFilter.filterSettings;
    const isImportantHighlighted = filters.isImportantHighlighted;
    const numberOfFiltersApplied = countFiltersApplied(filters);
    const categories = getActivityLogsCategories(containerEnum);
    const activityTypes = activityTypesFromReducer.map((activityType) => ({
        id: activityType,
        title: getActivityTypeLabel(activityType),
    }));

    const validTargetIds = release ? [...getAllTasks(release).map((task) => task.id), ...variables.map((v) => v.id ?? '')] : [];

    useEffect(() => {
        containerId && dispatch(init({ containerId, filterSettings, containerEnum }));
        return () => dispatch(reset());
    }, []);

    useKeyPress(
        { keys: 'Escape' },
        () => {
            if (isDrawerOpened) {
                handleDrawerClose();
            }
        },
        [isDrawerOpened],
    );

    const handleIntersect = () => {
        if (isLoading || isLastPage) return;
        dispatch(loadFilteredAndPagedLogs({ page: page + 1, logsFilter }));
    };

    const handleDrawerOpen = (event: MouseEvent<Element> | KeyboardEvent<Element>) => {
        if (isDrawerOpened) return;
        event.stopPropagation();
        dispatch(setIsDrawerOpened(true));
    };

    const handleDrawerClose = () => dispatch(setIsDrawerOpened(false));

    const handleClearAllFilters = () =>
        dispatch(
            loadFilteredAndPagedLogs({
                page: 0,
                logsFilter: {
                    containerId,
                    filterSettings: {
                        ...DEFAULT_FILTER_SETTINGS,
                        filter: logsFilter.filterSettings.filter,
                        isImportantHighlighted,
                        folderId: filterSettings.folderId,
                    },
                    containerEnum,
                },
            }),
        );

    const handleFilterChange = (filter: ActivityLogsFilterSettings) => {
        dispatch(
            loadFilteredAndPagedLogs({
                page: 0,
                logsFilter: { containerId, filterSettings: filter, containerEnum },
            }),
        );
    };
    const handleImportantHighlightedChange = () => dispatch(setHighlightImportant(!isImportantHighlighted));

    const handleDetailsClick = (dataId: string) => {
        dispatch(loadEventDialog(dataId));
    };

    const handleCloseDialog = () => {
        dispatch(closeEventDialog());
    };

    return (
        <div className="activity-logs-content">
            <div className="activity-logs-header">
                <DotTypography data-testid="activity-logs-title-typography" variant="h1">
                    History
                </DotTypography>
                <div className="activity-logs-header-filters">
                    {containerEnum !== ActivityLogsContainerEnum.TRIGGER && (
                        <DotSwitch checked={isImportantHighlighted} label="Highlight important" onChange={handleImportantHighlightedChange} />
                    )}
                    <DotSearch
                        className="activity-logs-filter"
                        id="activity-logs-filter"
                        onClear={() => handleFilterChange({ ...filters, filter: '' })}
                        onSearch={(searchText) => handleFilterChange({ ...filters, filter: searchText })}
                        placeholder="Filter by activity message, ID or user"
                        value={filters.filter}
                    />
                    <FilterButton numberOfFiltersApplied={numberOfFiltersApplied} onClick={handleDrawerOpen} />
                </div>
            </div>
            <ActivityLogsFilters
                categories={categories}
                dateFormat={dateFormat}
                filters={filters}
                numberOfFiltersApplied={numberOfFiltersApplied}
                onClearAll={handleClearAllFilters}
                onFilterChange={handleFilterChange}
                timeFormat={timeFormat}
                users={users}
            />
            <ActivityLogsTable
                containerEnum={containerEnum}
                dateFormat={dateFormat}
                dateOrder={filterSettings.dateAsc ? 'asc' : 'desc'}
                isFirstPage={page === 0}
                isImportantHighlighted={isImportantHighlighted}
                isLastPage={isLastPage}
                isLoading={isLoading}
                logs={logs}
                onDetailsClick={handleDetailsClick}
                onIntersect={handleIntersect}
                onUpdateTableData={(isDateAsc) => handleFilterChange({ ...filters, dateAsc: isDateAsc })}
                timeFormat={timeFormat}
                validTargetIds={validTargetIds}
            />
            <ActivityLogsFilterDrawer
                activityTypes={activityTypes}
                categories={categories}
                filter={filters}
                isDrawerOpened={isDrawerOpened}
                numberOfFiltersApplied={numberOfFiltersApplied}
                onClearAll={handleClearAllFilters}
                onDrawerClose={handleDrawerClose}
                onFilterChange={handleFilterChange}
                users={users}
            />
            {isEventDialogOpened && <ActivityLogsEventDialog content={eventDetails} onClose={handleCloseDialog} />}
        </div>
    );
};
