import React from 'react';
import { shallow } from 'enzyme';
import { DotSearch, DotSwitch, DotTypography } from '@digital-ai/dot-components';
import { act, mockIntersectionObserver, mockResizeObserver, mountWithStoreAndTheme, pressKey, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { ActivityLogsContainerEnum } from '../types';
import { ActivityLogs, ActivityLogsContent, ActivityLogsProps } from './activity-logs.component';
import { getActivityLogsCategories } from '../helper';
import { ActivityLogsFilterDrawer } from './activity-logs-filter-drawer.component';
import { mockTriggerActivityLogs } from '../__mocks__/activity-logs.mock';
import { ActivityLogsTable } from './table/activity-logs-table.component';
import { activityLogs as activityLogsSlice } from '../ducks/activity-logs.reducer';
import { ActivityLogsEventDialog } from './table/activity-logs-event-dialog.component';
import { ActionToolbarPortalContainer } from '@xlr-ui/app/features/main-navigation/action-toolbar/components/action-toolbar.portal';
import { releaseMock } from '@xlr-ui/app/features/tasks/__mocks__/release.mock';
import { ActivityLog, ActivityLogsFilterSettings } from '../../../types';
import { DEFAULT_FILTER_SETTINGS } from '../../../constants';
import { mockReleaseActivityLogs } from '../../../__mocks__/index.mock';
import { FilterButton } from '@xlr-ui/app/features/common/components/filter-button/filter-button.component';

const { reset, setIsDrawerOpened, setHighlightImportant, loadFilteredAndPagedLogs, closeEventDialog, loadEventDialog } = activityLogsSlice.actions;

describe('ActivityLogs component', () => {
    it('should render ActivityLogsContent component', () => {
        const wrapper = shallow(
            <ActivityLogs containerEnum={ActivityLogsContainerEnum.RELEASE} containerId="1" filterSettings={DEFAULT_FILTER_SETTINGS} release={releaseMock} />,
        );
        const activityLogsContentProps = wrapper.find(ActivityLogsContent).props();
        expect(activityLogsContentProps.containerEnum).toBe(ActivityLogsContainerEnum.RELEASE);
        expect(activityLogsContentProps.containerId).toBe('1');
        expect(activityLogsContentProps.release).toBe(releaseMock);
        expect(activityLogsContentProps.filterSettings).toBe(DEFAULT_FILTER_SETTINGS);
        const portalProps = wrapper.find(ActionToolbarPortalContainer).props();
        expect(portalProps.helpLink).toBe('concept/release-activity-logs');
    });
});

describe('ActivityLogsContent component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const observe = jest.fn();
    const unobserve = jest.fn();
    const stopPropagation = jest.fn();
    const dateFormat = 'y-MM-dd';
    const timeFormat = 'HH:mm';
    const activityLogsContainerEnum = ActivityLogsContainerEnum.RELEASE;
    const logs: ActivityLog[] = mockReleaseActivityLogs;
    const filterSettings: ActivityLogsFilterSettings = { ...DEFAULT_FILTER_SETTINGS, folderId: 'Folder1' };

    const defaultProps: ActivityLogsProps = {
        containerEnum: activityLogsContainerEnum,
        containerId: '1',
        filterSettings,
        release: releaseMock,
    };

    const defaultState = {
        activityLogs: {
            logs,
            logsFilter: {
                containerId: defaultProps.containerId,
                filterSettings: {
                    ...defaultProps.filterSettings,
                    filter: 'my text filter',
                    activityTypes: ['ATTACHMENT_ADDED_ON_TASK', 'COMMENT_ADDED'],
                },
            },
            activityTypes: ['ATTACHMENT_ADDED_ON_TASK', 'COMMENT_ADDED'],
            users: [
                {
                    username: 'admin',
                    fullName: 'Release Administrator',
                    email: 'releaseadmin@dai.com',
                },
            ],
            eventDetails: 'my trigger event details',
            isEventDialogOpened: true,
            isDrawerOpened: false,
            isLastPage: false,
            isLoading: false,
            page: 0,
            variables: [{ id: 'Variable1' }],
        },
        profile: {
            dateFormat,
            timeFormat,
        },
    };

    const getTitleTypography = () => wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'activity-logs-title-typography');
    const getDotSwitch = () => wrapper.find(DotSwitch);
    const getSearch = () => wrapper.findWhere((node) => node.is(DotSearch) && node.props().id === 'activity-logs-filter');
    const getFilterButton = () => wrapper.find(FilterButton);
    const getActivityLogsTable = () => wrapper.find(ActivityLogsTable);
    const getActivityLogsFilterDrawer = () => wrapper.find(ActivityLogsFilterDrawer);
    const getActivityLogsEventDialog = () => wrapper.find(ActivityLogsEventDialog);

    const mount = async (props: ActivityLogsProps = defaultProps, state = defaultState) => {
        await act(async () => {
            wrapper = mountWithStoreAndTheme(<ActivityLogsContent {...props} />, dispatch, state);
        });
    };

    beforeEach(() => {
        mockIntersectionObserver(observe, unobserve);
        mockResizeObserver();
    });

    afterEach(() => {
        jest.clearAllMocks();
    });

    it('should render properly', async () => {
        await mount();
        const titleTypography = getTitleTypography();
        expect(titleTypography).toExist();
        const titleTypographyProps = titleTypography.props();
        expect(titleTypographyProps.variant).toBe('h1');
        expect(titleTypographyProps.children).toBe('History');

        const dotSwitch = getDotSwitch();
        expect(dotSwitch).toExist();
        const dotSwitchProps = dotSwitch.props();
        expect(dotSwitchProps.checked).toBe(true);
        expect(dotSwitchProps.label).toBe('Highlight important');
        expect(dotSwitchProps.onChange).toStrictEqual(expect.any(Function));

        const search = getSearch();
        expect(search).toExist();
        const searchProps = search.props();
        expect(searchProps.placeholder).toBe('Filter by activity message, ID or user');

        const filterButton = getFilterButton();
        expect(filterButton).toExist();
        const filterButtonProps = filterButton.props();
        expect(filterButtonProps.onClick).toStrictEqual(expect.any(Function));
        expect(filterButtonProps.numberOfFiltersApplied).toBe(1);

        const activityLogsTable = getActivityLogsTable();
        expect(activityLogsTable).toExist();
        const activityLogsTableProps = activityLogsTable.props();
        expect(activityLogsTableProps.dateOrder).toBe('desc');
        expect(activityLogsTableProps.containerEnum).toBe(activityLogsContainerEnum);
        expect(activityLogsTableProps.isLastPage).toBe(false);
        expect(activityLogsTableProps.isImportantHighlighted).toBe(true);
        expect(activityLogsTableProps.isLoading).toBe(false);
        expect(activityLogsTableProps.validTargetIds).toStrictEqual(['Release1-Phase1-Task1', 'Variable1']);
        expect(activityLogsTableProps.logs).toStrictEqual(logs);
        expect(activityLogsTableProps.onIntersect).toStrictEqual(expect.any(Function));
        expect(activityLogsTableProps.onUpdateTableData).toStrictEqual(expect.any(Function));
        expect(activityLogsTableProps.dateFormat).toBe('YYYY-MM-DD');
        expect(activityLogsTableProps.timeFormat).toBe(timeFormat);

        const activityLogsFilterDrawer = getActivityLogsFilterDrawer();
        expect(activityLogsFilterDrawer).toExist();
        const activityLogsFilterDrawerProps = activityLogsFilterDrawer.props();
        expect(activityLogsFilterDrawerProps.activityTypes).toStrictEqual([
            {
                id: 'ATTACHMENT_ADDED_ON_TASK',
                title: 'Attachment added on task',
            },
            { id: 'COMMENT_ADDED', title: 'Comment added' },
        ]);
        expect(activityLogsFilterDrawerProps.categories).toStrictEqual(getActivityLogsCategories(defaultProps.containerEnum));
        expect(activityLogsFilterDrawerProps.filter).toStrictEqual({
            activityTypes: ['ATTACHMENT_ADDED_ON_TASK', 'COMMENT_ADDED'],
            comments: false,
            dateAsc: false,
            deliveryEdit: false,
            execution: false,
            filter: 'my text filter',
            folderId: 'Folder1',
            from: null,
            isImportantHighlighted: true,
            lifecycle: false,
            other: false,
            reassign: false,
            releaseEdit: false,
            reportingRecordEdit: false,
            security: false,
            taskEdit: false,
            to: null,
            triggerEdit: false,
            usernames: [],
        });
        expect(activityLogsFilterDrawerProps.isDrawerOpened).toBe(false);
        expect(activityLogsFilterDrawerProps.numberOfFiltersApplied).toBe(1);
        expect(activityLogsFilterDrawerProps.onClearAll).toStrictEqual(expect.any(Function));
        expect(activityLogsFilterDrawerProps.onDrawerClose).toStrictEqual(expect.any(Function));
        expect(activityLogsFilterDrawerProps.onFilterChange).toStrictEqual(expect.any(Function));
        expect(activityLogsFilterDrawerProps.users).toStrictEqual([
            {
                email: 'releaseadmin@dai.com',
                fullName: 'Release Administrator',
                username: 'admin',
            },
        ]);

        const dialogProps = getActivityLogsEventDialog().props();
        expect(dialogProps.content).toBe('my trigger event details');
        expect(dialogProps.onClose).toStrictEqual(expect.any(Function));
    });

    it('should NOT render Highlight important switch for trigger container', async () => {
        await mount(
            { ...defaultProps, containerEnum: ActivityLogsContainerEnum.TRIGGER },
            {
                ...defaultState,
                activityLogs: {
                    logs: mockTriggerActivityLogs,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        filterSettings: defaultProps.filterSettings,
                    },
                    activityTypes: [],
                    users: [],
                    eventDetails: 'my trigger event details',
                    isEventDialogOpened: false,
                    isDrawerOpened: false,
                    isLastPage: false,
                    isLoading: false,
                    page: 0,
                    variables: [],
                },
            },
        );
        const dotSwitch = getDotSwitch();
        expect(dotSwitch).not.toExist();
    });

    it("should render ActivityLogsTable component with dateOrder set to 'asc' when filterSettings.dateAsc is set to true true", () => {
        mount({ ...defaultProps, filterSettings: { ...filterSettings, dateAsc: true } });
        expect(getActivityLogsTable().props().dateOrder).toBe('asc');
    });

    it('should render ActivityLogsTable component with validTargetIds set to empty array when release is not defined', () => {
        mount({ ...defaultProps, release: undefined });
        expect(getActivityLogsTable().props().validTargetIds).toStrictEqual([]);
    });

    describe('dispatch actions', () => {
        it('should dispatch reset on unmount', async () => {
            await mount();
            wrapper.unmount();
            expect(dispatch).toHaveBeenCalledWith(reset());
        });

        it('should dispatch correct action when drawer is opened', async () => {
            await mount();
            const filterButton = getFilterButton();
            filterButton.invoke('onClick')({
                currentTarget: document.createElement('div'),
                stopPropagation,
            } as never);
            expect(dispatch).toHaveBeenCalledWith(setIsDrawerOpened(true));
        });

        it('should dispatch correct action when cleaning all filters', async () => {
            await mount();
            const activityLogsFilterDrawer = getActivityLogsFilterDrawer();
            activityLogsFilterDrawer.invoke('onClearAll')();
            expect(dispatch).toHaveBeenCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        filterSettings: { ...DEFAULT_FILTER_SETTINGS, filter: 'my text filter', folderId: filterSettings.folderId },
                        containerEnum: defaultProps.containerEnum,
                    },
                }),
            );
        });

        it('should dispatch correct action when changing filters', async () => {
            await mount();
            const activityLogsFilterDrawer = getActivityLogsFilterDrawer();
            activityLogsFilterDrawer.invoke('onFilterChange')({ comments: true } as never);
            expect(dispatch).toHaveBeenCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        containerEnum: defaultProps.containerEnum,
                        filterSettings: { comments: true } as never,
                    },
                }),
            );
        });

        it('should dispatch correct action when onChange handler has been executed on the filter input', async () => {
            await mount();
            const search = getSearch();
            search.invoke('onSearch')('new filter');
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        containerEnum: defaultProps.containerEnum,
                        filterSettings: expect.objectContaining({
                            filter: 'new filter',
                        }),
                    },
                }),
            );
        });

        it('should dispatch correct action when onClear handler has been executed on the filter input', async () => {
            await mount();
            const search = getSearch();
            search.invoke('onClear')();
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        containerEnum: defaultProps.containerEnum,
                        filterSettings: expect.objectContaining({
                            filter: '',
                        }),
                    },
                }),
            );
        });

        it('should dispatch correct action when onUpdateTableData is executed', async () => {
            await mount();
            const isDateAsc = true;
            const activityLogsTable = getActivityLogsTable();
            activityLogsTable.invoke('onUpdateTableData')(isDateAsc);
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        containerEnum: defaultProps.containerEnum,
                        filterSettings: expect.objectContaining({
                            dateAsc: isDateAsc,
                        }),
                    },
                }),
            );
        });

        it('should execute correct action when DotSwitch has been checked', async () => {
            await mount();
            const dotSwitch = getDotSwitch();
            dotSwitch.invoke('onChange')();
            expect(dispatch).toHaveBeenLastCalledWith(setHighlightImportant(false));
        });

        it('should dispatch correct actions for the event dialog', async () => {
            await mount();
            const activityLogsTable = getActivityLogsTable();
            activityLogsTable.invoke('onDetailsClick')('my data Id');
            expect(dispatch).toHaveBeenCalledWith(loadEventDialog('my data Id'));

            const dialog = getActivityLogsEventDialog();
            dialog.invoke('onClose')();
            expect(dispatch).toHaveBeenCalledWith(closeEventDialog());
        });

        it('should dispatch correct action when the drawer is closed', () => {
            mount();
            const activityLogsFilterDrawer = getActivityLogsFilterDrawer();
            activityLogsFilterDrawer.invoke('onDrawerClose')();
            expect(dispatch).toHaveBeenCalledWith(setIsDrawerOpened(false));
        });

        it('should dispatch close drawer action on Escape button press', () => {
            mount({ ...defaultProps }, { ...defaultState, activityLogs: { ...defaultState.activityLogs, isDrawerOpened: true } });
            pressKey('Escape');
            expect(dispatch).toHaveBeenLastCalledWith(setIsDrawerOpened(false));
        });

        it('should NOT dispatch an action when the onClick has been invoked on the DotIconButton but isDrawerOpened is set to true', () => {
            mount({ ...defaultProps }, { ...defaultState, activityLogs: { ...defaultState.activityLogs, isDrawerOpened: true } });
            const filterButton = getFilterButton();
            dispatch.mockReset();
            filterButton.invoke('onClick')({
                currentTarget: document.createElement('div'),
                stopPropagation,
            } as never);
            expect(dispatch).not.toHaveBeenCalled();
        });

        it('should dispatch correct action when onIntersect has been invoked', () => {
            mount();
            const activityLogsTable = getActivityLogsTable();
            activityLogsTable.invoke('onIntersect')();
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedLogs({
                    page: 1,
                    logsFilter: defaultState.activityLogs.logsFilter as never,
                }),
            );
        });

        it('should NOT dispatch an action when onIntersect has been invoked on the ActivityLogsTable but isLoading is set to true', () => {
            mount({ ...defaultProps }, { ...defaultState, activityLogs: { ...defaultState.activityLogs, isLoading: true } });
            const activityLogsTable = getActivityLogsTable();
            dispatch.mockReset();
            activityLogsTable.invoke('onIntersect')();
            expect(dispatch).not.toHaveBeenCalled();
        });

        it('should NOT dispatch an action when onIntersect has been invoked on the ActivityLogsTable but isLastPage is set to true', () => {
            mount({ ...defaultProps }, { ...defaultState, activityLogs: { ...defaultState.activityLogs, isLastPage: true } });
            const activityLogsTable = getActivityLogsTable();
            dispatch.mockReset();
            activityLogsTable.invoke('onIntersect')();
            expect(dispatch).not.toHaveBeenCalled();
        });
    });
});
