import React from 'react';
import { mountWithTheme, mountWithStoreAndTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { ActivityLogsFilterDrawer, ActivityLogsFilterDrawerProps } from './activity-logs-filter-drawer.component';
import { DotButton, DotClickAwayListener, DotDrawer, DotIconButton } from '@digital-ai/dot-components';
import { FilterGroup } from '@xlr-ui/app/features/common/components/filter/filter-group.component';
import { FilterAutocomplete } from './filters/filter-autocomplete.component';
import { FilterDate } from '@xlr-ui/app/features/common/components/filter/filter-date.component';
import { FilterUsername } from '@xlr-ui/app/features/common/components/filter/filter-username.component';
import { getActivityLogsCategories } from '../helper';
import { ActivityLogsContainerEnum } from '../types';
import { DEFAULT_FILTER_SETTINGS } from '../../../constants';

describe('ActivityLogsFilterDrawer component', () => {
    let wrapper: ReactWrapper;
    const onClearAll = jest.fn();
    const onDrawerClose = jest.fn();
    const onFilterChange = jest.fn();

    const defaultProps: ActivityLogsFilterDrawerProps = {
        activityTypes: [
            {
                id: '1',
                title: 'activityType1',
            },
        ],
        categories: getActivityLogsCategories(ActivityLogsContainerEnum.RELEASE),
        filter: {
            ...DEFAULT_FILTER_SETTINGS,
            activityTypes: ['1'],
            usernames: ['username1'],
            comments: true,
        },
        isDrawerOpened: true,
        numberOfFiltersApplied: 1,
        onClearAll,
        onDrawerClose,
        onFilterChange,
        users: [
            {
                username: 'username1',
            },
        ],
    };

    const getDotClickAwayListener = () => wrapper.find(DotClickAwayListener);
    const getDotDrawer = () => wrapper.find(DotDrawer);
    const getActivityLogsDrawerHeaderDiv = () => wrapper.find('div.activity-logs-drawer-header');
    const getCloseDrawerButton = () => getActivityLogsDrawerHeaderDiv().find(DotIconButton);
    const getClearAllButton = () => wrapper.find('.activity-logs-drawer-actions').find(DotButton);
    const getNumberOfFiltersApplied = () => wrapper.find('.activity-logs-drawer-actions h6');
    const getFilterGroup = (title: string) => wrapper.findWhere((node) => node.is(FilterGroup) && node.prop('title') === title);

    const mount = (props: ActivityLogsFilterDrawerProps = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<ActivityLogsFilterDrawer {...props} />, jest.fn(), { profile: { dateFormat: 'MM/dd/yyyy', timeFormat: 'HH:mm:ss' } });
    };

    it('should render properly', () => {
        mount();
        expect(getDotClickAwayListener()).toExist();
        const dotDrawer = getDotDrawer();
        expect(dotDrawer).toExist();
        const dotDrawerProps = dotDrawer.props();
        expect(dotDrawerProps.ModalProps).toStrictEqual({ hideBackdrop: true });
        expect(dotDrawerProps.PaperProps).toStrictEqual({
            style: {
                top: '104px',
                padding: '0',
                height: 'calc(100% - 104px)',
            },
        });
        expect(dotDrawerProps.anchor).toBe('right');
        expect(dotDrawerProps.className).toBe('activity-logs-drawer');
        expect(dotDrawerProps.open).toBe(true);
        expect(dotDrawerProps.variant).toBe('persistent');
        expect(dotDrawerProps.width).toBe('320px');

        const closeDrawerButton = getCloseDrawerButton();
        expect(closeDrawerButton).toExist();
        const closeDrawerButtonProps = closeDrawerButton.props();
        expect(closeDrawerButtonProps.iconId).toBe('close');

        const tooltipWrapper = mountWithTheme(closeDrawerButtonProps.tooltip);
        expect(tooltipWrapper.text()).toBe('CloseESC');

        closeDrawerButton.invoke('onClick')?.({} as never);
        expect(onDrawerClose).toHaveBeenCalledTimes(1);

        expect(getNumberOfFiltersApplied()).toIncludeText('Applied filters (1)');
    });

    it('should clear all the filters', () => {
        mount();
        getClearAllButton().invoke('onClick')?.('' as never);
        expect(onClearAll).toHaveBeenCalledTimes(1);
    });

    it('should show all filters by default and handle change', () => {
        mount();
        expect(wrapper.find(FilterGroup)).toHaveLength(5);

        const activityProps = getFilterGroup('Activity').find(FilterAutocomplete).props();
        expect(activityProps.inputId).toStrictEqual('activityTypes');
        expect(activityProps.options).toStrictEqual(defaultProps.activityTypes);
        expect(activityProps.placeholder).toStrictEqual('Filter by activity type');
        expect(activityProps.value).toStrictEqual([{ id: '1', title: 'activityType1' }]);
        activityProps.onChange(['2']);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, activityTypes: ['2'] });
        onFilterChange.mockReset();

        const usernamesProps = getFilterGroup('Performed by').find(FilterUsername).props();
        expect(usernamesProps.inputId).toStrictEqual('usernames');
        expect(usernamesProps.users).toStrictEqual(defaultProps.users);
        expect(usernamesProps.placeholder).toStrictEqual('Filter by user or system');
        expect(usernamesProps.value).toStrictEqual(['username1']);
        usernamesProps.onChange(['2']);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, usernames: ['2'] });
        onFilterChange.mockReset();

        const categoriesProps = getFilterGroup('Category').find(FilterAutocomplete).props();
        expect(categoriesProps.className).toStrictEqual('categories-autocomplete');
        expect(categoriesProps.inputId).toStrictEqual('categories');
        expect(categoriesProps.options).toStrictEqual(defaultProps.categories);
        expect(categoriesProps.placeholder).toStrictEqual('Filter by category');
        expect(categoriesProps.value).toStrictEqual([
            {
                id: 'comments',
                title: 'Comments',
            },
        ]);
        categoriesProps.onChange(['releaseEdit', 'comments']);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, comments: true, releaseEdit: true });
        onFilterChange.mockReset();

        const timeFromProps = getFilterGroup('Time from').find(FilterDate).props();
        expect(timeFromProps.inputId).toStrictEqual('time-from');
        expect(timeFromProps.value).toStrictEqual(defaultProps.filter.from);
        timeFromProps.onChange(123);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, from: 123 });
        onFilterChange.mockReset();

        const timeToProps = getFilterGroup('Time to').find(FilterDate).props();
        expect(timeToProps.inputId).toStrictEqual('time-to');
        expect(timeToProps.value).toStrictEqual(defaultProps.filter.to);
        timeToProps.onChange(123);
        expect(onFilterChange).toHaveBeenCalledWith({ ...defaultProps.filter, to: 123 });
        onFilterChange.mockReset();
    });
});
