package com.xebialabs.xlrelease.activity;

import java.util.Date;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.xlrelease.domain.ActivityLogEntry;
import com.xebialabs.xlrelease.domain.ReleaseActivity;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.service.UserInfoResolver;
import com.xebialabs.xlrelease.utils.Diff;

import static com.google.common.base.Objects.equal;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.xlrelease.activity.DurationFormatter.formatDuration;
import static com.xebialabs.xlrelease.activity.ReleaseActivityDateFormatter.formatDate;
import static com.xebialabs.xlrelease.activity.ReleaseActivityUserFormatter.quoteAssignee;
import static com.xebialabs.xlrelease.activity.ReleaseActivityUserFormatter.quoteUser;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_DESCRIPTION_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_DUE_DATE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_DURATION_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_FLAG_COMMENT_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_FLAG_STATUS_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_OWNER_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_SCHEDULED_START_DATE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_TAGS_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_TASK_TEAM_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_TITLE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_WAIT_FOR_SCHEDULED_START_DATE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_WATCHER_ADDED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_WATCHER_REMOVED;
import static scala.jdk.javaapi.CollectionConverters.asJava;
import static scala.jdk.javaapi.CollectionConverters.asScala;

@Component
public class DefaultTaskFieldsComparator implements TaskFieldsComparator<Task> {

    private UserInfoResolver userInfoResolver;

    @Autowired
    public DefaultTaskFieldsComparator(final UserInfoResolver userInfoResolver) {
        this.userInfoResolver = userInfoResolver;
    }

    @Override
    public Class<Task> getTaskClass() {
        return Task.class;
    }

    @Override
    public List<ActivityLogEntry> getLogs(Date timestamp, String username, final Task original, final Task updated) {
        List<ActivityLogEntry> logEntries = newArrayList();

        if (!equal(original.getTitle(), updated.getTitle())) {
            logEntries.add(TASK_TITLE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    original.getTitle(), updated.getTitle())
            );
        }

        if (!equal(original.getDescription(), updated.getDescription())) {
            logEntries.add(TASK_DESCRIPTION_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.getDescription(), updated.getDescription())
            );
        }

        if (!equal(original.getOwner(), updated.getOwner())) {
            logEntries.add(TASK_OWNER_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), quoteUser(original.getOwner(), userInfoResolver), quoteUser(updated.getOwner(), userInfoResolver))
            );
        }

        if (!equal(original.getTeam(), updated.getTeam())) {
            logEntries.add(TASK_TASK_TEAM_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), quoteAssignee(original.getTeam()), quoteAssignee(updated.getTeam()))
            );
        }

        if (!equal(original.getDueDate(), updated.getDueDate())) {
            logEntries.add(TASK_DUE_DATE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), formatDate(original.getDueDate()), formatDate(updated.getDueDate()))
            );
        }

        if (!equal(original.getScheduledStartDate(), updated.getScheduledStartDate())) {
            logEntries.add(TASK_SCHEDULED_START_DATE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), formatDate(original.getScheduledStartDate()), formatDate(updated.getScheduledStartDate()))
            );
        }

        if (!equal(original.isWaitForScheduledStartDate(), updated.isWaitForScheduledStartDate())) {
            logEntries.add(TASK_WAIT_FOR_SCHEDULED_START_DATE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), updated.isWaitForScheduledStartDate())
            );
        }

        if (!equal(original.getFlagStatus(), updated.getFlagStatus())) {
            logEntries.add(TASK_FLAG_STATUS_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.getFlagStatus().name(), updated.getFlagStatus().name())
            );
        }

        if (!equal(original.getFlagComment(), updated.getFlagComment())) {
            logEntries.add(TASK_FLAG_COMMENT_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.getFlagComment(), updated.getFlagComment())
            );
        }

        if (!equal(original.getPlannedDuration(), updated.getPlannedDuration())) {
            logEntries.add(TASK_DURATION_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), formatDuration(original.getPlannedDuration()), formatDuration(updated.getPlannedDuration()))
            );
        }

        if (!equal(original.getPrecondition(), updated.getPrecondition())) {
            logEntries.add(ReleaseActivity.TASK_PRECONDITION_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.getPrecondition(), updated.getPrecondition())
            );
        }

        if (!equal(original.isTaskFailureHandlerEnabled(), updated.isTaskFailureHandlerEnabled())) {
            logEntries.add(ReleaseActivity.TASK_FAILURE_HANDLE_ENABLED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.isTaskFailureHandlerEnabled(), updated.isTaskFailureHandlerEnabled())
            );
        }

        if (!equal(original.getTaskRecoverOp(), updated.getTaskRecoverOp())) {
            logEntries.add(ReleaseActivity.TASK_RECOVER_OP_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.getTaskRecoverOp(), updated.getTaskRecoverOp())
            );
        }

        if (!equal(original.getFailureHandler(), updated.getFailureHandler())) {
            logEntries.add(ReleaseActivity.TASK_RECOVER_SCRIPT_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.getFailureHandler(), updated.getFailureHandler())
            );
        }

        if (original.delayDuringBlackoutHasChanged(updated)) {
            logEntries.add(ReleaseActivity.TASK_POSTPONE_BLACKOUT_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.isDelayDuringBlackout(), updated.isDelayDuringBlackout())
            );
        }

        if (!equal(original.isCheckAttributes(), updated.isCheckAttributes())) {
            logEntries.add(ReleaseActivity.TASK_CHECK_ATTRIBUTES_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.isCheckAttributes(), updated.isCheckAttributes())
            );
        }

        if (!equal(newArrayListOrNull(original.getTags()), updated.getTags())) {
            logEntries.add(TASK_TAGS_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTitle(), original.getTags(), updated.getTags() != null ? newHashSet(updated.getTags()) : null)
            );
        }

        final Diff<String, String> watchersDiff = Diff.apply(
                asScala(original.getWatchers()).toIterable(),
                asScala(updated.getWatchers()).toIterable()
        );
        asJava(watchersDiff.deletedValues()).forEach(removed ->
                logEntries.add(TASK_WATCHER_REMOVED.create(timestamp, username, original.getType(), original.getId(), updated.getTitle(), removed))
        );
        asJava(watchersDiff.newValues()).forEach(added ->
                logEntries.add(TASK_WATCHER_ADDED.create(timestamp, username, original.getType(), original.getId(), updated.getTitle(), added))
        );

        return logEntries;
    }

    private List<String> newArrayListOrNull(List<String> tags) {
        return tags != null ? newArrayList(tags) : null;
    }
}
