import React from 'react';
import noop from 'lodash/noop';
import { DotButton, DotClickAwayListener, DotDrawer, DotIconButton, DotTypography } from '@digital-ai/dot-components';
import { User, WithTitleAndId } from '@xlr-ui/app/types';
import { getSelectedCategoryFromActivitySettings, getSelectedCategoryFromIds } from '../helper';
import { FilterGroup } from './filters/filter-group.component';
import { FilterAutocomplete } from './filters/filter-autocomplete.component';
import { FilterDate } from './filters/filter-date.component';
import { FilterUsername } from './filters/filter-username.component';
import './activity-logs-filter-drawer.component.less';
import { ActivityLogsFilterSettings } from '../../../types';

export interface ActivityLogsFilterDrawerProps {
    activityTypes: WithTitleAndId[];
    categories: WithTitleAndId[];
    filter: ActivityLogsFilterSettings;
    isDrawerOpened: boolean;
    numberOfFiltersApplied: number;
    onClearAll: () => void;
    onDrawerClose: () => void;
    onFilterChange: (filter: ActivityLogsFilterSettings) => void;
    users: User[];
}

export const ActivityLogsFilterDrawer = ({
    isDrawerOpened,
    numberOfFiltersApplied,
    onClearAll,
    onDrawerClose,
    onFilterChange,
    filter,
    categories,
    activityTypes,
    users,
}: ActivityLogsFilterDrawerProps) => {
    return (
        <DotClickAwayListener onClickAway={isDrawerOpened ? onDrawerClose : noop}>
            <div>
                <DotDrawer
                    ModalProps={{ hideBackdrop: true }}
                    PaperProps={{
                        style: {
                            top: '104px',
                            padding: '0',
                            height: 'calc(100% - 104px)',
                        },
                    }}
                    anchor="right"
                    className="activity-logs-drawer"
                    open={isDrawerOpened}
                    variant="persistent"
                    width="320px"
                >
                    {isDrawerOpened ? (
                        <>
                            <div className="activity-logs-drawer-header">
                                <DotTypography variant="h3">Filters</DotTypography>
                                <DotIconButton iconId="close" onClick={onDrawerClose} tooltip="Close" />
                            </div>
                            <div className="activity-logs-drawer-actions">
                                <DotTypography variant="subtitle2">{`Applied filters (${numberOfFiltersApplied})`}</DotTypography>
                                <DotButton onClick={onClearAll} type="text">
                                    Clear all
                                </DotButton>
                            </div>
                            <FilterGroup title="Activity">
                                <FilterAutocomplete
                                    inputId="activityTypes"
                                    onChange={(newActivityTypes) =>
                                        onFilterChange({
                                            ...filter,
                                            activityTypes: newActivityTypes,
                                        })
                                    }
                                    options={activityTypes}
                                    placeholder="Filter by activity type"
                                    value={activityTypes.filter((at) => filter.activityTypes.includes(at.id))}
                                />
                            </FilterGroup>
                            <FilterGroup title="Performed by">
                                <FilterUsername
                                    inputId="usernames"
                                    onChange={(usernames) => onFilterChange({ ...filter, usernames })}
                                    placeholder="Filter by user or system"
                                    users={users}
                                    value={filter.usernames}
                                />
                            </FilterGroup>
                            <FilterGroup title="Category">
                                <FilterAutocomplete
                                    className="categories-autocomplete"
                                    inputId="categories"
                                    onChange={(newCategories) => onFilterChange({ ...filter, ...getSelectedCategoryFromIds(newCategories) })}
                                    options={categories}
                                    placeholder="Filter by category"
                                    value={categories.filter((c) => getSelectedCategoryFromActivitySettings(filter).includes(c.id))}
                                />
                            </FilterGroup>
                            <FilterGroup title="Time from">
                                <FilterDate
                                    inputId="time-from"
                                    onChange={(from) =>
                                        onFilterChange({
                                            ...filter,
                                            from,
                                        })
                                    }
                                    value={filter.from}
                                />
                            </FilterGroup>
                            <FilterGroup title="Time to">
                                <FilterDate inputId="time-to" onChange={(to) => onFilterChange({ ...filter, to })} value={filter.to} />
                            </FilterGroup>
                        </>
                    ) : null}
                </DotDrawer>
            </div>
        </DotClickAwayListener>
    );
};
