package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.domain.ActivityCategory._
import com.xebialabs.xlrelease.domain.{ActivityCategory, ActivityLogEntry, ReleaseActivity}
import com.xebialabs.xlrelease.repository.sql.persistence.ListExtensions
import com.xebialabs.xlrelease.repository.{ActivityLogRepository, ReleaseRepository}
import com.xebialabs.xlrelease.views.LogsFilters
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.Pageable
import org.springframework.stereotype.Service

import java.util
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

@Service
class ReleaseActivityLogsService @Autowired()(userInfoResolver: UserInfoResolver,
                                              activityLogRepository: ActivityLogRepository,
                                              val releaseRepository: ReleaseRepository,
                                              val archivedActivityLogsService: ArchivedActivityLogsService
                                             ) extends BaseActivityLogsService(userInfoResolver, activityLogRepository) {
  private lazy val releaseActivityOps = ReleaseActivity.values().toList.asJava

  override def withCategories(logEntry: ActivityLogEntry, filters: LogsFilters): Boolean = {
    val allowedCategories = ListBuffer.empty[ActivityCategory]
    if (filters.withReleaseEdit) allowedCategories += RELEASE_EDIT
    if (filters.withTaskEdit) allowedCategories += TASK_EDIT
    if (filters.withComments) allowedCategories += COMMENTS
    if (filters.withLifecycle) allowedCategories += LIFECYCLE
    if (filters.withReassign) allowedCategories += REASSIGN
    if (filters.withReportingRecordEdit) allowedCategories += REPORTING_RECORD_EDIT
    if (filters.withSecurity) allowedCategories += SECURITY
    if (filters.withOther) allowedCategories += OTHER

    val activityCategories = ReleaseActivity.safeValueOf(logEntry.getActivityType).getCategories.asScala

    // if we don't select any category, we search all
    if (allowedCategories.isEmpty) {
      allowedCategories += IMPORTANT
      allowedCategories ++= activityCategories
    }
    allowedCategories exists activityCategories
  }

  override def fetchLogs(containerId: String, logsFilters: LogsFilters, pageable: Pageable): List[ActivityLogEntry] = {
    if (releaseRepository.exists(containerId)) {
      activityLogRepository.findAllLogsOf(containerId, logsFilters, releaseActivityOps, pageable).asScala.toList
    } else if (archivedActivityLogsService.exists(containerId)) {
      filterLogs(
        archivedActivityLogsService.getActivityLogs(containerId).asScala.toList,
        logsFilters,
        pageable)
    } else {
      throw new NotFoundException(s"Release [${containerId}] does not exist in the repository or archive")
    }
  }

  override def getUniqueUsers(containerId: String): util.List[String] = {
    if (releaseRepository.exists(containerId)) {
      activityLogRepository.findUniqueUsers(containerId)
    } else if (archivedActivityLogsService.exists(containerId)) {
      val activityLogs = archivedActivityLogsService.getActivityLogs(containerId).asScala.toList
      val distinctUsers = activityLogs.map(_.getUsername).distinct.asJavaMutable()
      if (distinctUsers.removeIf(_ == null) && !distinctUsers.contains("SYSTEM")) {
        distinctUsers.add("SYSTEM")
      }
      distinctUsers
    } else {
      throw new NotFoundException(s"Release [${containerId}] does not exist in the repository or archive")
    }
  }

  override def getUniqueActivityTypes(containerId: String): util.List[String] = {
    if (releaseRepository.exists(containerId)) {
      activityLogRepository.findUniqueActivityTypes(containerId)
    } else if (archivedActivityLogsService.exists(containerId)) {
      val activityLogs = archivedActivityLogsService.getActivityLogs(containerId).asScala.toList
      activityLogs.map(_.getActivityType).distinct.asJava
    } else {
      throw new NotFoundException(s"Release [${containerId}] does not exist in the repository or archive")
    }
  }
}
