import moment from 'moment';
import {
    checkIfLastPage,
    checkIfMessageIsMarkdown,
    getActivityLogUserTitle,
    getActivityTypeLabel,
    getFilterSettingsForLoadPayload,
    getFiltersWithLogUser,
    groupActivityLogsByDate,
} from '.';
import { ActivityLog } from '../types';
import { LOG_USER, SYSTEM_USER } from '../constants';
import { mockReleaseActivityLogs } from '../__mocks__/index.mock';
import { User } from '@xlr-ui/app/types';

describe('common activity helpers', () => {
    describe('getFilterSettingsForLoadPayload', () => {
        it('should return filters without fields for payload', () => {
            expect(getFilterSettingsForLoadPayload({ fakeKey: 'fakeValue', folderId: 'Folder1', isImportantHighlighted: true } as never)).toStrictEqual({
                fakeKey: 'fakeValue',
            });
        });
    });

    describe('checkIfLastPage', () => {
        const logs: ActivityLog[] = [
            {
                activityType: 'STAGE_STARTED',
            },
            {
                activityType: 'TRIGGER_ENABLED',
            },
        ] as never;

        test('should return true if logs array is empty', () => {
            const result = checkIfLastPage([], 10);
            expect(result).toBe(true);
        });

        test('should return true if logs length is less than logsPerPage', () => {
            const result = checkIfLastPage(logs, 5);
            expect(result).toBe(true);
        });

        test('should return false if logs length is equal to logsPerPage', () => {
            const result = checkIfLastPage(logs, 2);
            expect(result).toBe(false);
        });

        test('should return false if logs length is greater than logsPerPage', () => {
            const result = checkIfLastPage(logs, 1);
            expect(result).toBe(false);
        });
    });

    describe('getFiltersWithLogUser', () => {
        it('should add LOG_USER if SYSTEM_USER is present', () => {
            const logsFilter = {
                filterSettings: {
                    usernames: [SYSTEM_USER],
                },
            } as never;

            const result = getFiltersWithLogUser(logsFilter);
            expect(result.filterSettings.usernames).toEqual([SYSTEM_USER, LOG_USER]);
        });

        it('should not modify usernames if SYSTEM_USER is not present', () => {
            const logsFilter = {
                filterSettings: {
                    usernames: ['admin'],
                },
            } as never;

            const result = getFiltersWithLogUser(logsFilter);
            expect(result.filterSettings.usernames).toEqual(['admin']);
        });
    });

    describe('getActivityTypeLabel', () => {
        it('should format activity type correctly', () => {
            expect(getActivityTypeLabel('STAGE_STARTED')).toBe('Stage started');
            expect(getActivityTypeLabel('TRIGGER_ENABLED')).toBe('Trigger enabled');
            expect(getActivityTypeLabel('RELEASE_COMPLETED')).toBe('Release completed');
        });
    });

    describe('checkIfMessageIsMarkdown', () => {
        it('should return true for TRIGGER_EXECUTED with a release link in the message', () => {
            const result = checkIfMessageIsMarkdown('TRIGGER_EXECUTED', 'Check this link: #/releases/123');
            expect(result).toBe(true);
        });

        it('should return true for TRIGGER_SKIPPED with a template link in the message', () => {
            const result = checkIfMessageIsMarkdown('TRIGGER_SKIPPED', 'Check this link: #/templates/123');
            expect(result).toBe(true);
        });

        it('should return false for TRIGGER_EXECUTED without a release or template link in the message', () => {
            const result = checkIfMessageIsMarkdown('TRIGGER_EXECUTED', 'No relevant link here');
            expect(result).toBe(false);
        });

        it('should return false for an activity type other than TRIGGER_EXECUTED or TRIGGER_SKIPPED', () => {
            const result = checkIfMessageIsMarkdown('OTHER_TYPE', 'Check this link: #/releases/123');
            expect(result).toBe(false);
        });

        it('should return false for TRIGGER_SKIPPED without a release or template link in the message', () => {
            const result = checkIfMessageIsMarkdown('TRIGGER_SKIPPED', 'No relevant link here');
            expect(result).toBe(false);
        });
    });

    describe('groupActivityLogsByDate', () => {
        it('should groups logs by same day', () => {
            const baseTime = moment('2024-04-18T10:30:00Z').valueOf();

            const logs: ActivityLog[] = [
                {
                    ...mockReleaseActivityLogs[0],
                    eventTime: baseTime,
                },
                {
                    ...mockReleaseActivityLogs[1],
                    eventTime: baseTime + 1000 * 60 * 60, // same day, 1 hour later
                },
            ];

            const result = groupActivityLogsByDate(logs);

            const expectedKey = moment(baseTime).startOf('day').valueOf().toString();

            expect(Object.keys(result)).toHaveLength(1);
            expect(result[expectedKey]).toHaveLength(2);
            expect(result[expectedKey][0].activityType).toBe('RELEASE_CREATED');
            expect(result[expectedKey][1].activityType).toBe('RELEASE_STARTED');
        });

        it('should groups logs by different days', () => {
            const time1 = moment('2024-04-16T00:01:00Z').valueOf();
            const time2 = moment('2024-04-19T00:01:00Z').valueOf();

            const logs: ActivityLog[] = [
                {
                    ...mockReleaseActivityLogs[0],
                    eventTime: time1,
                },
                {
                    ...mockReleaseActivityLogs[1],
                    eventTime: time2,
                },
            ];

            const result = groupActivityLogsByDate(logs);

            const keys = Object.keys(result);
            expect(keys).toHaveLength(2);
            expect(result[keys[0]]).toHaveLength(1);
            expect(result[keys[1]]).toHaveLength(1);
        });

        it('should return empty object for empty input', () => {
            const result = groupActivityLogsByDate([]);
            expect(result).toEqual({});
        });
    });

    describe('getActivityLogUserTitle', () => {
        it('should return SYSTEM when user is SYSTEM', () => {
            const user: User = { username: SYSTEM_USER, fullName: 'System' };
            expect(getActivityLogUserTitle(user)).toBe(SYSTEM_USER);
        });

        it('should return SYSTEM when user is LOG_USER', () => {
            const user: User = { username: LOG_USER, fullName: 'Logger' };
            expect(getActivityLogUserTitle(user)).toBe(SYSTEM_USER);
        });

        it('should return fullName when user is not system and has fullName', () => {
            const user: User = { username: 'user123', fullName: 'Test User' };
            expect(getActivityLogUserTitle(user)).toBe('Test User');
        });

        it('should return username when user has no fullName', () => {
            const user: User = { username: 'user123' };
            expect(getActivityLogUserTitle(user)).toBe('user123');
        });
    });
});
