import React from 'react';
import moment from 'moment';
import { DotButton } from '@digital-ai/dot-components';
import { getActivityTypeLabel } from '../../helper';
import { ActivityLogsFilterSettings } from '../../types';
import { User, WithTitleAndId } from '@xlr-ui/app/types';
import { FilterGroupChips } from './filter-group-chips.component';
import { UserChip } from './user-chip.component';
import './activity-logs-filters.component.less';

export interface ActivityLogsFiltersProps {
    categories: WithTitleAndId[];
    dateFormat: string;
    filters: ActivityLogsFilterSettings;
    numberOfFiltersApplied: number;
    onClearAll: () => void;
    onFilterChange: (filters: ActivityLogsFilterSettings) => void;
    timeFormat: string;
    users: User[];
}

export const ActivityLogsFilters = ({
    categories,
    dateFormat,
    filters,
    numberOfFiltersApplied,
    onClearAll,
    onFilterChange,
    timeFormat,
    users,
}: ActivityLogsFiltersProps) => {
    const categoriesFiltersApplied: WithTitleAndId[] = categories.filter((category) => filters[category.id as keyof ActivityLogsFilterSettings]);
    const usersFiltersApplied: WithTitleAndId[] = users
        .filter((user) => filters.usernames.includes(user.username))
        .map((user) => ({
            id: user.username,
            title: user.fullName || user.username,
        }));
    const fromFilterApplied: WithTitleAndId[] = filters.from
        ? [
              {
                  id: filters.from.toString(),
                  title: `${moment(filters.from).format(dateFormat)} - ${moment(filters.from).format(timeFormat)}`,
              },
          ]
        : [];
    const toFilterApplied: WithTitleAndId[] = filters.to
        ? [
              {
                  id: filters.to.toString(),
                  title: `${moment(filters.to).format(dateFormat)} - ${moment(filters.to).format(timeFormat)}`,
              },
          ]
        : [];

    return (
        <div className="activity-logs-filters">
            {filters.activityTypes.length > 0 && (
                <FilterGroupChips
                    onDelete={(id) =>
                        onFilterChange({
                            ...filters,
                            activityTypes: filters.activityTypes.filter((a) => a !== id),
                        })
                    }
                    title="Activity:"
                    values={filters.activityTypes.map((at) => ({ id: at, title: getActivityTypeLabel(at) }))}
                />
            )}
            {usersFiltersApplied.length > 0 && (
                <FilterGroupChips title="Performed by:" values={usersFiltersApplied}>
                    {(value) => (
                        <UserChip
                            key={value.id}
                            onDelete={(userId) =>
                                onFilterChange({
                                    ...filters,
                                    usernames: filters.usernames.filter((u) => u !== userId),
                                })
                            }
                            value={value}
                        />
                    )}
                </FilterGroupChips>
            )}
            {categoriesFiltersApplied.length > 0 && (
                <FilterGroupChips
                    onDelete={(id) =>
                        onFilterChange({
                            ...filters,
                            [id]: false,
                        })
                    }
                    title="Categories:"
                    values={categoriesFiltersApplied}
                />
            )}
            {fromFilterApplied.length > 0 && (
                <FilterGroupChips
                    onDelete={() =>
                        onFilterChange({
                            ...filters,
                            from: null,
                        })
                    }
                    title="From:"
                    values={fromFilterApplied}
                />
            )}
            {toFilterApplied.length > 0 && (
                <FilterGroupChips
                    onDelete={() =>
                        onFilterChange({
                            ...filters,
                            to: null,
                        })
                    }
                    title="To:"
                    values={toFilterApplied}
                />
            )}
            {numberOfFiltersApplied > 0 && (
                <DotButton onClick={onClearAll} type="text">
                    Clear all
                </DotButton>
            )}
        </div>
    );
};
