import { activityLogs, initialState, LoadFilteredAndPagedLogsAction, LogsFilter } from './activity-logs.reducer';
import { ActivityLogsContainerEnum, ActivityLogsFilterSettings } from '../types';
import { DEFAULT_FILTER_SETTINGS } from '../constants';

const {
    closeEventDialog,
    init,
    loadEventDialog,
    loadFilteredAndPagedLogs,
    reset,
    setActivityTypes,
    setHighlightImportant,
    setEventDetails,
    setEventDialogOpened,
    setIsDrawerOpened,
    setIsLastPage,
    setIsLoading,
    setLogs,
    setLogsFilter,
    setPage,
    setUsers,
    setVariables,
} = activityLogs.actions;

describe('activity logs reducer', () => {
    it('should have initial state', () => {
        expect(activityLogs.name).toBe('activityLogs');
        // if a new reducer is added, this test should fail and dev must fix and add new one
        expect(Object.keys(activityLogs.actions)).toStrictEqual([
            'init',
            'loadFilteredAndPagedLogs',
            'setLogsFilter',
            'setLogs',
            'setHighlightImportant',
            'setPage',
            'setIsLastPage',
            'setIsLoading',
            'setIsDrawerOpened',
            'setActivityTypes',
            'setUsers',
            'loadEventDialog',
            'closeEventDialog',
            'setEventDialogOpened',
            'setEventDetails',
            'setVariables',
            'reset',
        ]);
        expect(activityLogs.reducer(undefined, { type: 'default' })).toStrictEqual(initialState);
    });

    describe('reducers', () => {
        describe('init', () => {
            it('should do nothing with state', () => {
                const mockPayload = {
                    containerEnum: ActivityLogsContainerEnum.RELEASE,
                    containerId: '123',
                    filterSettings: {
                        comments: true,
                    } as ActivityLogsFilterSettings,
                };
                const state = activityLogs.reducer(initialState, init(mockPayload));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('loadFilteredAndPagedLogs', () => {
            it('should do nothing with state', () => {
                const mockPayload = {
                    page: 2,
                } as LoadFilteredAndPagedLogsAction;
                const state = activityLogs.reducer(initialState, loadFilteredAndPagedLogs(mockPayload));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('setLogsFilter', () => {
            it('should update state', () => {
                const mockPayload = {
                    containerEnum: ActivityLogsContainerEnum.RELEASE,
                    containerId: '123',
                    filterSettings: DEFAULT_FILTER_SETTINGS,
                    page: 2,
                } as LogsFilter;
                const state = activityLogs.reducer(initialState, setLogsFilter(mockPayload as never));
                expect(state).toStrictEqual({
                    ...initialState,
                    logsFilter: {
                        ...mockPayload,
                    },
                });
            });
        });

        describe('setLogs', () => {
            it('should update state', () => {
                const mockPayload = [{ test: 'test' }];
                const state = activityLogs.reducer(initialState, setLogs(mockPayload as never));
                expect(state).toStrictEqual({
                    ...initialState,
                    logs: mockPayload,
                });
            });
        });

        describe('setHighlightImportant', () => {
            it('should do nothing with state', () => {
                const state = activityLogs.reducer(initialState, setHighlightImportant(true));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('setPage', () => {
            it('should update state', () => {
                const page = 2;
                const state = activityLogs.reducer(initialState, setPage(page));
                expect(state).toStrictEqual({
                    ...initialState,
                    page,
                });
            });
        });

        describe('setIsLastPage', () => {
            it('should update state', () => {
                const state = activityLogs.reducer(initialState, setIsLastPage(true));
                expect(state).toStrictEqual({
                    ...initialState,
                    isLastPage: true,
                });
            });
        });

        describe('setIsLoading', () => {
            it('should update state', () => {
                const state = activityLogs.reducer(initialState, setIsLoading(true));
                expect(state).toStrictEqual({
                    ...initialState,
                    isLoading: true,
                });
            });
        });

        describe('setIsDrawerOpened', () => {
            it('should update state', () => {
                const state = activityLogs.reducer(initialState, setIsDrawerOpened(true));
                expect(state).toStrictEqual({
                    ...initialState,
                    isDrawerOpened: true,
                });
            });
        });

        describe('setActivityTypes', () => {
            it('should update state', () => {
                const state = activityLogs.reducer(initialState, setActivityTypes(['a', 'b']));
                expect(state).toStrictEqual({
                    ...initialState,
                    activityTypes: ['a', 'b'],
                });
            });
        });

        describe('setUsers', () => {
            it('should update state', () => {
                const state = activityLogs.reducer(initialState, setUsers([{ id: '1', name: 'test' }] as never));
                expect(state).toStrictEqual({
                    ...initialState,
                    users: [{ id: '1', name: 'test' }],
                });
            });
        });

        describe('loadEventDialog', () => {
            it('should do nothing with state', () => {
                const state = activityLogs.reducer(initialState, loadEventDialog('myDataId'));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('closeEventDialog', () => {
            it('should update state', () => {
                const state = activityLogs.reducer({ ...initialState, eventDetails: 'somethting', isEventDialogOpened: true }, closeEventDialog());
                expect(state.isEventDialogOpened).toStrictEqual(false);
                expect(state.eventDetails).toBeUndefined();
            });
        });

        describe('setEventDialogOpened', () => {
            it('should update state', () => {
                const state = activityLogs.reducer(initialState, setEventDialogOpened(true));
                expect(state.isEventDialogOpened).toStrictEqual(true);
            });
        });

        describe('setEventDetails', () => {
            it('should update state', () => {
                const state = activityLogs.reducer(initialState, setEventDetails('my details'));
                expect(state.eventDetails).toStrictEqual('my details');
            });
        });

        describe('setVariables', () => {
            it('should update state', () => {
                const state = activityLogs.reducer(initialState, setVariables([{ id: '1', name: 'test' }] as never));
                expect(state.variables).toStrictEqual([{ id: '1', name: 'test' }]);
            });
        });

        describe('reset', () => {
            it('should update state', () => {
                const state = activityLogs.reducer(initialState, reset());
                expect(state).toStrictEqual({
                    ...initialState,
                    isDrawerOpened: false,
                    logs: [],
                });
            });
        });
    });
});
