import React from 'react';
import { DotSkeleton, DotTable, Order, TableRowProps } from '@digital-ai/dot-components';
import classNames from 'classnames';
import { ActivityLog, ActivityLogsContainerEnum } from '../../types';
import { LOGS_ITEMS_PER_PAGE } from '../../constants';
import { useTableInfiniteScrollPagination } from '@xlr-ui/app/features/common/hooks/table-infinite-scroll-pagination.hook';
import { checkIfImportantActivity, getTableColumns } from '../../helper';
import { ActivityLogsUser } from './activity-logs-user.component';
import { ActivityLogsActivity } from './activity-logs-activity.component';
import { ActivityLogsTypeRelease } from './activity-logs-type.release.component';
import { ActivityLogsTypeDelivery } from './activity-logs-type.delivery.component';
import { ActivityLogsTime } from './activity-logs-time.component';
import { ActivityLogsDetails } from './activity-logs-details.component';
import './activity-logs-table.component.less';

export interface ActivityLogsTableProps {
    containerEnum: ActivityLogsContainerEnum;
    dateFormat: string;
    dateOrder: Order;
    isFirstPage: boolean;
    isImportantHighlighted: boolean;
    isLastPage: boolean;
    isLoading: boolean;
    logs: ActivityLog[];
    onDetailsClick: (dataId: string) => void;
    onIntersect: () => void;
    onUpdateTableData: (isDateAsc: boolean) => void;
    timeFormat: string;
    validTargetIds: string[];
}

export const NUMBER_OF_LOADING_ROWS = 3;

export const ActivityLogsTable = ({
    containerEnum,
    dateFormat,
    dateOrder,
    isFirstPage,
    isImportantHighlighted,
    isLastPage,
    isLoading,
    logs,
    onDetailsClick,
    onIntersect,
    onUpdateTableData,
    timeFormat,
    validTargetIds,
}: ActivityLogsTableProps) => {
    const { observerTarget } = useTableInfiniteScrollPagination<ActivityLog, HTMLDivElement>({
        isFirstPage,
        items: logs,
        onIntersect,
    });

    const getSkeletonTableRow = (index: number, isObserverRow: boolean) => {
        return {
            id: `loading-row-${index}`,
            rowData: {
                name: (
                    <div ref={isObserverRow ? observerTarget : undefined}>
                        <DotSkeleton />
                    </div>
                ),
                activityType: <DotSkeleton />,
                targetType: <DotSkeleton />,
                eventTime: <DotSkeleton />,
            },
            className: 'loading-row',
        };
    };

    const getActivityLogsTypeComponent = (currentActivityLog: ActivityLog) => {
        switch (containerEnum) {
            case ActivityLogsContainerEnum.DELIVERY:
                return <ActivityLogsTypeDelivery targetId={currentActivityLog.targetId} targetType={currentActivityLog.targetType} />;
            case ActivityLogsContainerEnum.RELEASE:
                return (
                    <ActivityLogsTypeRelease
                        targetId={currentActivityLog.targetId}
                        targetType={currentActivityLog.targetType}
                        validTargetIds={validTargetIds}
                    />
                );
            default:
                return <></>;
        }
    };

    const getTableRows = () => {
        if (isLoading && !logs.length) {
            return [...Array(LOGS_ITEMS_PER_PAGE)].map((_, index: number) => getSkeletonTableRow(index, false));
        } else if (logs.length) {
            const tableRows: TableRowProps[] = logs.map((currentActivityLog, index) => {
                return {
                    id: `${index}`,
                    rowData: {
                        name: <ActivityLogsUser user={currentActivityLog.user} />,
                        message: (
                            <ActivityLogsActivity
                                activityType={currentActivityLog.activityType}
                                eventTime={currentActivityLog.eventTime}
                                message={currentActivityLog.message}
                            />
                        ),
                        details: <ActivityLogsDetails dataId={currentActivityLog.dataId} onDetailsClick={onDetailsClick} />,
                        targetType: getActivityLogsTypeComponent(currentActivityLog),
                        eventTime: <ActivityLogsTime dateFormat={dateFormat} eventTime={currentActivityLog.eventTime} timeFormat={timeFormat} />,
                    },
                    className: classNames({ 'important-activity': isImportantHighlighted && checkIfImportantActivity(currentActivityLog.activityType) }),
                };
            });
            !isLastPage &&
                tableRows.push(
                    ...Array(NUMBER_OF_LOADING_ROWS)
                        .fill(undefined)
                        .map((_, index: number) => {
                            const isObserverRow = index === 0;
                            return getSkeletonTableRow(index, isObserverRow);
                        }),
                );
            return tableRows;
        }

        return [];
    };

    const handleUpdateTableData = (order: Order) => {
        onUpdateTableData(order === 'asc');
    };

    const tableRowsData = getTableRows();

    return (
        <DotTable
            className="activity-logs-table"
            columns={getTableColumns(containerEnum)}
            data={tableRowsData}
            onUpdateData={handleUpdateTableData}
            order={dateOrder}
            orderBy="eventTime"
        />
    );
};
