package com.xebialabs.xlrelease.activity;

import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.xlrelease.domain.ActivityLogEntry;
import com.xebialabs.xlrelease.domain.CreateReleaseTask;
import com.xebialabs.xlrelease.domain.variables.Variable;

import static com.google.common.base.Objects.equal;
import static com.xebialabs.deployit.util.PasswordEncrypter.getInstance;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_OUTPUT_PROPERTIES_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_RELEASE_PASSWORD_VARIABLE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_RELEASE_TAGS_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_RELEASE_TEMPLATE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_RELEASE_TITLE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_RELEASE_VARIABLE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_START_RELEASE_FLAG_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TASK_WAIT_FOR_RELEASE_FLAG_UPDATED;
import static java.util.stream.Stream.concat;

@Component
public class CreateReleaseTaskFieldsComparator implements TaskFieldsComparator<CreateReleaseTask> {

    private DefaultTaskFieldsComparator defaultTaskFieldsComparator;

    @Autowired
    public CreateReleaseTaskFieldsComparator(final DefaultTaskFieldsComparator defaultTaskFieldsComparator) {
        this.defaultTaskFieldsComparator = defaultTaskFieldsComparator;
    }

    @Override
    public Class<CreateReleaseTask> getTaskClass() {
        return CreateReleaseTask.class;
    }

    @Override
    public List<ActivityLogEntry> getLogs(Date timestamp, String username, final CreateReleaseTask original, final CreateReleaseTask updated) {
        List<ActivityLogEntry> updateLogs = defaultTaskFieldsComparator.getLogs(timestamp, username, original, updated);

        if (!equal(original.getNewReleaseTitle(), updated.getNewReleaseTitle())) {
            updateLogs.add(TASK_RELEASE_TITLE_UPDATED.create(timestamp, username, original.getTaskType(), original.getId(),
                    updated.getTitle(), original.getNewReleaseTitle(), updated.getNewReleaseTitle())
            );
        }

        if (!equal(original.getStartRelease(), updated.getStartRelease())) {
            updateLogs.add(TASK_START_RELEASE_FLAG_UPDATED.create(timestamp, username, original.getTaskType(), original.getId(),
                    updated.getTitle(), original.getStartRelease(), updated.getStartRelease())
            );
        }

        if (!equal(original.getWaitForRelease(), updated.getWaitForRelease())) {
            updateLogs.add(TASK_WAIT_FOR_RELEASE_FLAG_UPDATED.create(timestamp, username, original.getTaskType(), original.getId(),
                    updated.getTitle(), original.getWaitForRelease(), updated.getWaitForRelease())
            );
        }

        if (!equal(original.getTemplateId(), updated.getTemplateId())) {
            updateLogs.add(TASK_RELEASE_TEMPLATE_UPDATED.create(timestamp, username, original.getTaskType(), original.getId(),
                    updated.getTitle(), original.getTemplateId(), updated.getTemplateId())
            );
        } else {
            addVariableUpdateLogs(timestamp, username, updateLogs, original, updated);
        }

        if (!equal(original.getCreatedReleaseId(), updated.getCreatedReleaseId())) {
            updateLogs.add(TASK_OUTPUT_PROPERTIES_UPDATED.create(timestamp, username, original.getTaskType(), original.getId(),
                    updated.getTitle(), "Created release ID", original.getCreatedReleaseId(), updated.getCreatedReleaseId())
            );
        }

        if (!Objects.equals(original.getReleaseTags(), updated.getReleaseTags())) {
            if (!Objects.equals(new HashSet<>(original.getReleaseTags()), new HashSet<>(updated.getReleaseTags()))) {
                updateLogs.add(TASK_RELEASE_TAGS_UPDATED.create(timestamp, username, original.getTaskType(), original.getId(),
                        updated.getTitle(), original.getReleaseTags(), updated.getReleaseTags())
                );
            }
        }

        return updateLogs;
    }

    private void addVariableUpdateLogs(Date timestamp, String username,
                                       final List<ActivityLogEntry> updateLogs, final CreateReleaseTask original, final CreateReleaseTask updated) {
        concat(original.getTemplateVariables().stream(), updated.getTemplateVariables().stream())
                .map(Variable::getKey)
                .distinct()
                .forEach(key -> {
                    String oldValue = getVariableValue(original.getTemplateVariables(), key);
                    String newValue = getVariableValue(updated.getTemplateVariables(), key);
                    boolean isOldPassword = isPasswordVariable(original.getTemplateVariables(), key);
                    boolean isNewPassword = isPasswordVariable(updated.getTemplateVariables(), key);
                    final boolean isVariablePassword = isOldPassword || isNewPassword;

                    if (isVariablePassword && !equal(getInstance().ensureDecrypted(oldValue), getInstance().ensureDecrypted(newValue))) {
                        updateLogs.add(TASK_RELEASE_PASSWORD_VARIABLE_UPDATED.create(timestamp, username, original.getTaskType(), original.getId(),
                                updated.getTitle(), key)
                        );
                    } else if (!isVariablePassword && !equal(oldValue, newValue)) {
                        updateLogs.add(TASK_RELEASE_VARIABLE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                                updated.getTitle(), key, oldValue, newValue)
                        );
                    }
                });
    }

    private String getVariableValue(List<Variable> variables, String key) {
        return variables.stream().filter(v -> v.getKey().equals(key)).findFirst().map(Variable::getValueAsString).orElse("");
    }

    private boolean isPasswordVariable(List<Variable> variables, String key) {
        return variables.stream().filter(v -> v.getKey().equals(key)).findFirst().map(Variable::isPassword).orElse(false);
    }
}
