import { all, call, put, select, takeLatest } from 'redux-saga/effects';
import { PayloadAction } from '@reduxjs/toolkit';
import { SagaIterator } from 'redux-saga';
import { httpGET, httpPOST } from '@xlr-ui/app/features/common/services/http';
import { activityLogs, ActivityLogsState, getActivityLogsState, LoadFilteredAndPagedLogsAction, LogsFilter } from './activity-logs.reducer';
import { ActivityLogsContainerEnum } from '../types';
import { LOGS_ITEMS_PER_PAGE } from '../constants';
import { User, Variable } from '@xlr-ui/app/types';
import { getReleaseVariables } from '@xlr-ui/app/features/tasks/ducks/variables-service.saga';
import { ActivityLog } from '../../../types';
import { checkIfLastPage, getFilterSettingsForLoadPayload, getFiltersWithLogUser } from '../../../helper';
import { LOG_USER, SYSTEM_USER } from '../../../constants';
import { updateClientSettingsAndUrl } from '../../../ducks/saga';

const {
    init,
    loadEventDialog,
    loadFilteredAndPagedLogs,
    setActivityTypes,
    setEventDetails,
    setEventDialogOpened,
    setHighlightImportant,
    setIsLastPage,
    setIsLoading,
    setLogs,
    setLogsFilter,
    setPage,
    setUsers,
    setVariables,
} = activityLogs.actions;

export function* initSaga(action: PayloadAction<LogsFilter>): SagaIterator {
    try {
        const page = 0;
        const logsFilter = action.payload;
        yield put(setIsLoading(true));
        yield put(setPage(page));
        if (logsFilter.containerEnum === ActivityLogsContainerEnum.RELEASE) {
            const variables: Array<Variable> = yield call(getReleaseVariables, logsFilter.containerId);
            yield put(setVariables(variables));
        }
        const [{ data: activityTypes }, { data: users }] = yield all([
            call(httpGET, `activity/${logsFilter.containerId}/logs/activity-types`),
            call(httpGET, `activity/${logsFilter.containerId}/logs/users`),
        ]);
        const usersWithoutTaskComment = replaceLogUserWithSystemUser(users);
        yield put(setActivityTypes(activityTypes));
        yield put(setUsers(usersWithoutTaskComment));

        const cleanedLogFilters = cleanLogFilters(logsFilter, activityTypes, usersWithoutTaskComment);
        yield put(setLogsFilter(cleanedLogFilters));
        const logs = yield call(loadLogs, page, cleanedLogFilters);

        const isLastPage = checkIfLastPage(logs, LOGS_ITEMS_PER_PAGE);
        yield put(setIsLastPage(isLastPage));
        yield put(setLogs(logs));
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* loadFilteredAndPagedLogsSaga(action: PayloadAction<LoadFilteredAndPagedLogsAction>) {
    try {
        const { logsFilter, page } = action.payload;
        const { logs: oldLogs }: ActivityLogsState = yield select(getActivityLogsState);
        yield put(setIsLoading(true));
        if (page === 0 && oldLogs.length > 0) {
            yield put(setLogs([]));
        }
        yield put(setPage(page));
        yield put(setLogsFilter(logsFilter));
        const logs: ActivityLog[] = yield call(loadLogs, page, logsFilter);
        let newLogs: ActivityLog[] = oldLogs;

        if (page === 0) {
            newLogs = logs;
        } else if (logs.length > 0) {
            newLogs = oldLogs.concat(logs);
        }

        const isLastPage = checkIfLastPage(logs, LOGS_ITEMS_PER_PAGE);
        yield put(setIsLastPage(isLastPage));
        yield put(setLogs(newLogs));
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* loadEventDialogSaga(action: PayloadAction<string>) {
    yield put(setEventDialogOpened(true));
    const { data } = yield call(httpGET, `activity/data/${action.payload}`);
    if (data.content) {
        // eslint-disable-next-line angular/json-functions
        data.content = JSON.parse(data.content);
    }
    // eslint-disable-next-line angular/json-functions
    yield put(setEventDetails(JSON.stringify(data, null, 2)));
}

export function* loadLogs(page: number, logsFilter: LogsFilter) {
    const logsFilterWithTaskComment = getFiltersWithLogUser(logsFilter);
    const { data: logs } = yield call(
        httpPOST,
        `activity/${logsFilter.containerId}/logs/search?page=${page}&resultsPerPage=${LOGS_ITEMS_PER_PAGE}`,
        getFilterSettingsForLoadPayload(logsFilterWithTaskComment.filterSettings),
    );
    // we don't store the logsFilterWithTaskComment in the url since it might contain the task comment user
    yield call(updateClientSettingsAndUrl, logsFilter.containerId, logsFilter.filterSettings);
    return logs;
}

export function* setHighlightImportantSaga(action: PayloadAction<boolean>) {
    const { logsFilter }: ActivityLogsState = yield select(getActivityLogsState);
    const newFilterSettings = { ...logsFilter.filterSettings, isImportantHighlighted: action.payload };
    yield put(setLogsFilter({ ...logsFilter, filterSettings: newFilterSettings }));
    yield call(updateClientSettingsAndUrl, logsFilter.containerId, newFilterSettings);
}

export function cleanLogFilters(logsFilter: LogsFilter, activityTypes: string[], users: User[]) {
    // when manually typing to url some value that is not present in filters drawer, then remove those invalid filters
    return {
        ...logsFilter,
        filterSettings: {
            ...logsFilter.filterSettings,
            usernames: logsFilter.filterSettings.usernames.filter((username) => users.some((u) => u.username === username)),
            activityTypes: logsFilter.filterSettings.activityTypes.filter((activityType) => activityTypes.includes(activityType)),
        },
    };
}

export function replaceLogUserWithSystemUser(users: User[]): User[] {
    const hasLogUser = users.some((user) => user.username === LOG_USER);
    if (hasLogUser) {
        const hasSystemUser = users.some((user) => user.username === SYSTEM_USER);
        if (hasSystemUser) {
            // remove log user if system user is present, since we are only showing system user
            return users.filter((user) => user.username !== LOG_USER);
        } else {
            // replace log user with system user
            return users.map((user) => (user.username === LOG_USER ? { username: SYSTEM_USER } : user));
        }
    }
    return users;
}

export default function* activityLogsSaga() {
    yield all([
        takeLatest(init, initSaga),
        takeLatest(loadFilteredAndPagedLogs, loadFilteredAndPagedLogsSaga),
        takeLatest(setHighlightImportant, setHighlightImportantSaga),
        takeLatest(loadEventDialog, loadEventDialogSaga),
    ]);
}
