package com.xebialabs.xlrelease.activity

import com.xebialabs.xlrelease.domain.ReleaseActivity._
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.ActivityLogRepository
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@EventListener
class PhaseActivityLoggingEventHandler @Autowired()(val eventBus: XLReleaseEventBus, val activityLogRepository: ActivityLogRepository)
  extends ActivityLogger[PhaseEvent] {

  @Subscribe
  def onEvent(event: PhaseEvent): Unit = log(event)

  override def logEntries: PartialFunction[XLReleaseEvent, LoggingParams] = {
    onPhaseCUDEvent orElse onPhaseExecutionEvent
  }

  private val onPhaseCUDEvent: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev: PhaseCreatedEvent =>
      LoggingParams(releaseIdFrom(ev.phase), Option(ev.username),
        PHASE_CREATED.create(ev, ev.phase, ev.phase.getTitle)
      )

    case ev: PhaseDeletedEvent =>
      LoggingParams(releaseIdFrom(ev.phase), Option(ev.username),
        PHASE_DELETED.create(ev, ev.phase, ev.phase.getTitle)
      )

    case ev: PhaseUpdatedEvent =>
      val logs = new PhaseFieldsComparator(ev.timestamp, ev.username, ev.original, ev.updated).getLogs
      LoggingParams(releaseIdFrom(ev.updated), Option(ev.username), logs)

    case ev: PhaseMovedEvent =>
      LoggingParams(releaseIdFrom(ev.movedPhase), Option(ev.username),
        PHASE_MOVED.create(ev, ev.movedPhase, ev.movedPhase.getTitle)
      )

    case ev: PhaseDuplicatedEvent =>
      LoggingParams(releaseIdFrom(ev.phaseDuplicate), Option(ev.username),
        PHASE_DUPLICATED.create(ev, ev.phaseDuplicate, ev.phaseDuplicate.getTitle)
      )
  }

  private val onPhaseExecutionEvent: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev: PhaseStartedEvent =>
      LoggingParams(releaseIdFrom(ev.phase), Option(ev.username),
        PHASE_STARTED.create(ev, ev.phase, ev.phase.getTitle)
      )

    case ev: PhaseFailedEvent =>
      LoggingParams(releaseIdFrom(ev.phase), Option(ev.username),
        PHASE_FAILED.create(ev, ev.phase, ev.phase.getTitle)
      )

    case ev: PhaseStartedFailingEvent =>
      LoggingParams(releaseIdFrom(ev.phase), Option(ev.username),
        PHASE_FAILING.create(ev, ev.phase, ev.phase.getTitle)
      )

    case ev: PhaseRetriedEvent =>
      LoggingParams(releaseIdFrom(ev.phase), Option(ev.username),
        PHASE_RESTARTED.create(ev, ev.phase, ev.phase.getTitle)
      )

    case ev: PhaseCompletedEvent =>
      LoggingParams(releaseIdFrom(ev.phase), Option(ev.username),
        PHASE_COMPLETED.create(ev, ev.phase, ev.phase.getTitle)
      )

    case ev: PhaseClosedEvent =>
      LoggingParams(releaseIdFrom(ev.phase), Option(ev.username),
        PHASE_CLOSED.create(ev, ev.phase, ev.phase.getTitle)
      )
  }

}
