package com.xebialabs.xlrelease.domain;

import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlrelease.activity.ActivityOps;
import com.xebialabs.xlrelease.domain.events.XLReleaseEvent;

import static com.google.common.collect.Sets.newLinkedHashSet;
import static com.xebialabs.xlrelease.domain.ActivityCategory.COMMENTS;
import static com.xebialabs.xlrelease.domain.ActivityCategory.IMPORTANT;
import static com.xebialabs.xlrelease.domain.ActivityCategory.LIFECYCLE;
import static com.xebialabs.xlrelease.domain.ActivityCategory.REASSIGN;
import static com.xebialabs.xlrelease.domain.ActivityCategory.RELEASE_EDIT;
import static com.xebialabs.xlrelease.domain.ActivityCategory.REPORTING_RECORD_EDIT;
import static com.xebialabs.xlrelease.domain.ActivityCategory.SECURITY;
import static com.xebialabs.xlrelease.domain.ActivityCategory.TASK_EDIT;

public enum ReleaseActivity implements ActivityOps {

    TEMPLATE_CREATED("Created template", LIFECYCLE),
    TEMPLATE_CREATED_FROM_AS_CODE("Created template from as-code %s", LIFECYCLE),
    TEMPLATE_UPDATED_FROM_AS_CODE("Updated template from as-code %s", RELEASE_EDIT),
    TEMPLATE_DUPLICATED("Duplicated template to '%s'", RELEASE_EDIT),
    TEMPLATE_MOVED("Moved template from folder '%s' to folder '%s'", RELEASE_EDIT),
    TEMPLATE_CATEGORY_UPDATED("Updated categories to:\n%s", RELEASE_EDIT),
    TEMPLATE_AUTHOR_UPDATED("Changed author from '%s' to '%s'", RELEASE_EDIT),

    RELEASE_TITLE_UPDATED("Changed title from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_DESCRIPTION_UPDATED("Changed description from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_DUE_DATE_UPDATED("Changed due date from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_SCHEDULED_START_DATE_UPDATED("Changed scheduled start date from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_OWNER_UPDATED("Changed owner from %s to %s", REASSIGN, RELEASE_EDIT), // no single quotes here since it uses ReleaseActivityUserFormatter
    RELEASE_TAGS_UPDATED("Updated tags to:\n%s", RELEASE_EDIT),
    RELEASE_FLAG_STATUS_UPDATED("Changed flag status from '%s' to '%s'", RELEASE_EDIT, IMPORTANT),
    RELEASE_FLAG_COMMENT_UPDATED("Changed flag comment from '%s' to '%s'", RELEASE_EDIT, IMPORTANT, COMMENTS),
    RELEASE_ABORT_RELEASE_ON_FAILURE_UPDATED("Changed 'Abort release on failure' from '%s' to '%s'", RELEASE_EDIT),
    WORKFLOW_ABORT_WORKFLOW_ON_FAILURE_UPDATED("Changed 'Abort Workflow execution on failure' from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_ALLOW_PASSWORDS_IN_ALL_FIELDS_ON_FAILURE_UPDATED("Changed 'Allow passwords in all fields' from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_DISABLE_NOTIFICATIONS_UPDATED("Changed 'Disable notifications' from '%s' to '%s'", RELEASE_EDIT),

    COMMENT_ADDED("Added comment '%s' on task '%s'", TASK_EDIT, COMMENTS),
    COMMENT_UPDATED("Updated comment '%s' to '%s' on task '%s'", TASK_EDIT, COMMENTS),
    COMMENT_DELETED("Deleted comment '%s' on task '%s'", TASK_EDIT, COMMENTS),

    PHASE_CREATED("Created phase '%s'", RELEASE_EDIT),
    PHASE_RENAMED("Renamed phase '%s' to '%s'", RELEASE_EDIT),
    PHASE_DESCRIPTION_UPDATED("Changed phase '%s' description from '%s' to '%s'", RELEASE_EDIT),
    PHASE_DURATION_UPDATED("Changed phase '%s' duration from '%s' to '%s'", RELEASE_EDIT),
    PHASE_DUE_DATE_UPDATED("Changed phase '%s' due date from '%s' to '%s'", RELEASE_EDIT),
    PHASE_SCHEDULED_START_DATE_UPDATED("Changed phase '%s' scheduled start date from '%s' to '%s'", RELEASE_EDIT),
    PHASE_COLOR_CHANGED("Changed phase '%s' color from '%s' to '%s'", RELEASE_EDIT),
    PHASE_MOVED("Moved phase '%s'", RELEASE_EDIT),
    PHASE_DELETED("Deleted phase '%s'", RELEASE_EDIT),
    PHASE_DUPLICATED("Duplicated phase to '%s'", RELEASE_EDIT),
    PHASE_STARTED("Started phase '%s'", LIFECYCLE, IMPORTANT),
    PHASE_FAILED("Failed phase '%s'", LIFECYCLE),
    PHASE_FAILING("Phase '%s' started failing", LIFECYCLE),
    PHASE_RESTARTED("Restarted phase '%s'", LIFECYCLE),
    PHASE_COMPLETED("Completed phase '%s'", LIFECYCLE),
    PHASE_CLOSED("Closed phase '%s' and skipped all its tasks", LIFECYCLE),

    TASK_CREATED("Created task '%s' of type '%s'", RELEASE_EDIT, TASK_EDIT),
    TASK_MOVED_BETWEEN_CONTAINERS("Moved task '%s' from '%s' to '%s'", RELEASE_EDIT),
    TASK_MOVED_WITHIN_CONTAINER("Moved task '%s' within '%s'", RELEASE_EDIT),
    TASK_DELETED("Deleted task '%s'", RELEASE_EDIT, IMPORTANT),
    TASK_TITLE_UPDATED("Changed task title from '%s' to '%s'", TASK_EDIT),
    TASK_DESCRIPTION_UPDATED("Changed task '%s' description from '%s' to '%s'", TASK_EDIT),
    TASK_DURATION_UPDATED("Changed task '%s' duration from '%s' to '%s'", TASK_EDIT),
    TASK_OWNER_UPDATED("Changed task '%s' owner from %s to %s", TASK_EDIT, REASSIGN), // no single quotes here since it uses ReleaseActivityUserFormatter
    TASK_TASK_TEAM_UPDATED("Changed task '%s' team from %s to %s", TASK_EDIT, REASSIGN), // no single quotes here since it uses ReleaseActivityUserFormatter
    TASK_DUE_DATE_UPDATED("Changed task '%s' due date from '%s' to '%s'", TASK_EDIT),
    TASK_SCHEDULED_START_DATE_UPDATED("Changed task '%s' scheduled start date from '%s' to '%s'", TASK_EDIT),
    TASK_WAIT_FOR_SCHEDULED_START_DATE_UPDATED("Changed task '%s' wait for scheduled start date to '%s'", TASK_EDIT),
    TASK_FLAG_STATUS_UPDATED("Flagged task '%s' from '%s' to '%s'", TASK_EDIT, IMPORTANT),
    TASK_FLAG_COMMENT_UPDATED("Changed task '%s' flag status from '%s' to '%s'", TASK_EDIT, IMPORTANT, COMMENTS),
    TASK_COPIED("Copied task to '%s'", RELEASE_EDIT),
    TASK_COPIED_FROM("Copied task '%s' from %s '%s'", RELEASE_EDIT),
    TASK_INPUT_VARIABLES_UPDATED("Changed task '%s' variable list from '%s' to '%s'", TASK_EDIT),
    TASK_NOTIFICATION_ADDRESSES_UPDATED("Changed task '%s' notification email addresses from '%s' to '%s'", TASK_EDIT),
    TASK_NOTIFICATION_CC_UPDATED("Changed task '%s' notification email CC list from '%s' to '%s'", TASK_EDIT),
    TASK_NOTIFICATION_BCC_UPDATED("Changed task '%s' notification email BCC list from '%s' to '%s'", TASK_EDIT),
    TASK_NOTIFICATION_REPLY_TO_UPDATED("Changed task '%s' notification email reply-to address from '%s' to '%s'", TASK_EDIT),
    TASK_NOTIFICATION_FROM_UPDATED("Changed task '%s' notification email sender address from '%s' to '%s'", TASK_EDIT),
    TASK_NOTIFICATION_SENDER_USERNAME_UPDATED("Changed task '%s' notification email sender username from '%s' to '%s'", TASK_EDIT),
    TASK_NOTIFICATION_SENDER_PASSWORD_UPDATED("Changed task '%s' notification email sender password", TASK_EDIT),
    TASK_NOTIFICATION_PRIORITY_UPDATED("Changed task '%s' notification email priority from '%s' to '%s'", TASK_EDIT),
    TASK_NOTIFICATION_SUBJECT_UPDATED("Changed task '%s' notification subject from '%s' to '%s'", TASK_EDIT),
    TASK_NOTIFICATION_BODY_UPDATED("Changed task '%s' notification email body from '%s' to '%s'", TASK_EDIT),
    TASK_SCRIPT_UPDATED("Changed task '%s' script from '%s' to '%s'", TASK_EDIT),
    TASK_INPUT_PROPERTY_UPDATED("Changed task '%s' input property '%s' from '%s' to '%s'", TASK_EDIT),
    TASK_INPUT_PROPERTY_PASSWORD_UPDATED("Changed task '%s' input property '%s'", TASK_EDIT),
    TASK_OUTPUT_PROPERTIES_UPDATED("Changed task '%s' output property '%s' from '%s' to '%s'", TASK_EDIT),
    TASK_PRECONDITION_UPDATED("Changed task '%s' precondition from '%s' to '%s'", TASK_EDIT),
    TASK_TYPE_CHANGED("Changed task '%s' type from '%s' to '%s'", TASK_EDIT),
    TASK_RELEASE_TITLE_UPDATED("Changed task '%s' release title from '%s' to '%s'", TASK_EDIT),
    TASK_RELEASE_TEMPLATE_UPDATED("Changed task '%s' release template from '%s' to '%s'", TASK_EDIT),
    TASK_START_RELEASE_FLAG_UPDATED("Changed task '%s' start release flag from '%s' to '%s'", TASK_EDIT),
    TASK_WAIT_FOR_RELEASE_FLAG_UPDATED("Changed task '%s' wait on release to finish flag from '%s' to '%s'", TASK_EDIT),
    TASK_RELEASE_VARIABLE_UPDATED("Changed task '%s' variable '%s' from '%s' to '%s'", TASK_EDIT),
    TASK_RELEASE_PASSWORD_VARIABLE_UPDATED("Changed task '%s' password variable '%s'", TASK_EDIT),
    TASK_RELEASE_TAGS_UPDATED("Changed task '%s' release tags from '%s' to '%s'", TASK_EDIT),
    // deprecated, if removed from here then we can't search for them in database level, to be deleted in 26.1 ?
    TASKS_LOCKED("Locked task(s) '%s'", TASK_EDIT),
    TASKS_UNLOCKED("Unlocked task(s) '%s'", TASK_EDIT),
    ////
    TASK_LOCKED("Locked task '%s'", TASK_EDIT),
    TASK_UNLOCKED("Unlocked task '%s'", TASK_EDIT),
    TASK_FAILURE_HANDLE_ENABLED("Changed task '%s' failure handler enabled from '%s' to '%s'", TASK_EDIT),
    TASK_RECOVER_OP_UPDATED("Changed task '%s' recovery operation from '%s' to '%s'", TASK_EDIT),
    TASK_RECOVER_SCRIPT_UPDATED("Changed task '%s' recovery script from '%s' to '%s'", TASK_EDIT),
    TASK_POSTPONE_BLACKOUT_UPDATED("Changed task '%s' 'Postpone during blackout' from '%s' to '%s'", TASK_EDIT),
    TASK_CHECK_ATTRIBUTES_UPDATED("Changed task '%s' property 'Check environment availability' from '%s' to '%s'", TASK_EDIT),
    TASK_TAGS_UPDATED("Changed task '%s' tags from '%s' to '%s'", TASK_EDIT),
    TEAM_CREATED("Created team '%s'", SECURITY, RELEASE_EDIT),
    TEAM_UPDATED("Updated team '%s' with members '%s' and roles '%s'", REASSIGN, SECURITY),
    TEAM_DELETED("Removed team '%s'", SECURITY, RELEASE_EDIT),
    FOLDER_TEAM_MERGED("Merged teams and permissions from template '%s' into folder '%s'", SECURITY, RELEASE_EDIT),
    REMOVE_TEMPLATE_TEAMS("Removed teams from template %s. It will inherit teams and permissions from folder %s.", SECURITY, RELEASE_EDIT),
    TASK_WATCHER_ADDED("Changed task '%s' watchers, added '%s'", TASK_EDIT),
    TASK_WATCHER_REMOVED("Changed task '%s' watchers, removed '%s'", TASK_EDIT),

    GATE_CONDITION_CREATED("Created gate task '%s' condition '%s'", TASK_EDIT),
    GATE_CONDITION_TITLE_UPDATED("Updated gate task '%s' condition title from '%s' to '%s'", TASK_EDIT),
    GATE_CONDITION_FULFILLED("Fulfilled gate task '%s' condition '%s'", TASK_EDIT, LIFECYCLE),
    GATE_CONDITION_UNFULFILLED("Unfulfilled gate task '%s' condition '%s'", TASK_EDIT, LIFECYCLE),
    GATE_CONDITION_DELETED("Deleted gate task '%s' condition '%s'", TASK_EDIT),
    GATE_FAIL_DEPENDENCY_FLAG_UPDATED("Changed 'Fail if any dependency fails' flag of Task '%s' from '%s' to '%s'", TASK_EDIT),

    LINK_ADDED("Created link on group '%s' from task '%s' to task '%s'", TASK_EDIT),
    LINK_REMOVED("Removed link on group '%s' from task '%s' to task '%s'", TASK_EDIT),

    DEPENDENCY_CREATED("Added Dependency in gate '%s' on '%s'", TASK_EDIT),
    DEPENDENCY_UPDATED("Changed Dependency in gate '%s' on '%s' to '%s'", TASK_EDIT),
    DEPENDENCY_DELETED("Deleted Dependency in gate '%s' on '%s'", TASK_EDIT),

    // this event should be in trigger module but it belongs to a release, so it needs to be here when searching for it
    RELEASE_CREATED_FROM_TRIGGER("Created release '%s' from trigger '%s' and template '%s'", LIFECYCLE),
    RELEASE_CREATED_FROM_TEMPLATE("Created release '%s' from template '%s'", LIFECYCLE),
    WORKFLOW_EXECUTION_CREATED_FROM_TEMPLATE("Created workflow execution '%s' from template '%s'", LIFECYCLE),
    RELEASE_CREATED_FROM_CREATE_RELEASE_TASK("Created release '%s' from create release task '%s' and template '%s'", LIFECYCLE),
    RELEASE_CREATED_FROM_DSL("Created release '%s' from Groovy (DSL)", LIFECYCLE),
    WORKFLOW_EXECUTION_CREATED_FROM_DSL("Created workflow execution '%s' from Groovy (DSL)", LIFECYCLE),
    WORKFLOW_TEMPLATE_CREATED_FROM_DSL("Created workflow template '%s' from Groovy (DSL)", LIFECYCLE),
    RELEASE_RESTORED_FROM_REVISION("Restored template '%s' from revision '%s'", LIFECYCLE),
    RELEASE_CREATED("Created empty release '%s'", LIFECYCLE, RELEASE_EDIT),
    RELEASE_STARTED("Started release", LIFECYCLE, IMPORTANT),
    WORKFLOW_EXECUTION_STARTED("Started workflow execution", LIFECYCLE, IMPORTANT),
    RELEASE_STARTED_FROM_CREATE_RELEASE_TASK("Started release '%s' from create release task '%s'", LIFECYCLE),
    RELEASE_FAILED("Failed release", LIFECYCLE),
    WORKFLOW_EXECUTION_FAILED("Failed workflow execution", LIFECYCLE),
    RELEASE_FAILING("Release started failing", LIFECYCLE),
    WORKFLOW_EXECUTION_FAILING("Workflow execution started failing", LIFECYCLE),
    RELEASE_RESTARTED("Restarted release", LIFECYCLE),
    WORKFLOW_EXECUTION_RESTARTED("Restarted workflow execution", LIFECYCLE),
    RELEASE_COMPLETED("Completed release", LIFECYCLE, IMPORTANT),
    WORKFLOW_EXECUTION_COMPLETED("Completed workflow execution", LIFECYCLE, IMPORTANT),
    RELEASE_ABORTED("Aborted release with comment '%s'", LIFECYCLE, IMPORTANT),
    WORKFLOW_EXECUTION_ABORTED("Aborted workflow execution with comment '%s'", LIFECYCLE, IMPORTANT),

    TASK_STARTED("Started task '%s'", LIFECYCLE),
    TASK_DELAYED("Activated task '%s' with scheduled start date '%s'", LIFECYCLE),
    TASK_DELAYED_DUE_TO_BLACKOUT("Activated task '%s' with scheduled start date '%s' due to a blackout", LIFECYCLE),
    TASK_COMPLETED("Completed task '%s'", LIFECYCLE),
    TASK_COMPLETED_IN_ADVANCE("Completed in advance task '%s'", LIFECYCLE),
    TASK_SKIPPED("Skipped task '%s'", LIFECYCLE, IMPORTANT),
    TASK_SKIPPED_IN_ADVANCE("Skipped in advance task '%s'", LIFECYCLE, IMPORTANT),
    TASK_FAILED("Failed task '%s', reason: %s", LIFECYCLE, IMPORTANT),
    TASK_RESTARTED("Restarted task '%s'", LIFECYCLE),
    TASK_FAILING("Task '%s' started failing", LIFECYCLE),
    TASK_REOPENED("Reopened task '%s'", LIFECYCLE),
    TASK_RECOVERY_STARTED("Task '%s' started recovering. Recovery operation: '%s', failure handler script is %s.", LIFECYCLE, IMPORTANT),
    TASK_RECOVERED("Task '%s' recovery complete. Recovery operation: '%s', failure handler script was %s.", LIFECYCLE, IMPORTANT),
    TASK_ABORT_SCRIPT_STARTED("Abort script of task '%s' started. Abort script is %s.", LIFECYCLE, IMPORTANT),
    TASK_ABORT_SCRIPT_COMPLETED("Abort script of task '%s' completed. Abort script is %s.", LIFECYCLE, IMPORTANT),
    TASK_WAITING_FOR_INPUT("Task '%s' requires input for variables: %s", LIFECYCLE),
    TASK_FACET_CREATED("Created attribute on task '%s' of type '%s'", TASK_EDIT),
    TASK_FACET_UPDATED("Updated attribute on task '%s' of type '%s'", TASK_EDIT),
    TASK_FACET_DELETED("Deleted attribute on task '%s' of type '%s'", TASK_EDIT),
    TEMPLATE_IMPORTED("Imported template", RELEASE_EDIT),

    PERMISSIONS_UPDATED("Updated permissions to:\n%s", SECURITY),

    ATTACHMENT_ADDED("Added attachment '%s' (%s)", RELEASE_EDIT),
    ATTACHMENT_ADDED_ON_TASK("Added attachment '%s' (%s) on task '%s'", RELEASE_EDIT),
    ATTACHMENT_DELETED("Deleted attachment '%s'", RELEASE_EDIT),
    ATTACHMENT_DELETED_FROM_TASK("Deleted attachment '%s' on task '%s'", RELEASE_EDIT),

    TEMPLATE_LOGO_UPDATED("Updated logo: '%s' (%s)", RELEASE_EDIT),

    TEMPLATE_ALLOW_CONCURRENT_RELEASES_FROM_TRIGGER_UPDATED("Changed 'Allow concurrent triggered releases' from '%s' to '%s'", RELEASE_EDIT),

    RELEASE_VARIABLE_CREATED("Created variable '%s'", RELEASE_EDIT),
    RELEASE_VARIABLE_DELETED("Deleted variable '%s'", RELEASE_EDIT),
    RELEASE_VARIABLE_REPLACED("Replaced variable '%s' with '%s'", RELEASE_EDIT),
    RELEASE_VARIABLE_RENAMED("Renamed variable '%s' to '%s'", RELEASE_EDIT),

    RELEASE_VARIABLE_VALUE_UPDATED("Changed variable '%s' value from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_VARIABLE_PASSWORD_VALUE_UPDATED("Changed password variable '%s'", RELEASE_EDIT),
    RELEASE_VARIABLE_NAME_UPDATED("Changed variable '%s' name from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_VARIABLE_LABEL_UPDATED("Changed variable '%s' label from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_VARIABLE_DESCRIPTION_UPDATED("Changed variable '%s' description from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_VARIABLE_REQUIRED_UPDATED("Changed variable '%s' required flag from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_VARIABLE_SHOW_ON_CREATE_UPDATED("Changed variable '%s' 'show on create release form' flag from '%s' to '%s'", RELEASE_EDIT),

    GLOBAL_VARIABLE_CREATED("Created global variable '%s'", RELEASE_EDIT),
    GLOBAL_VARIABLE_VALUE_UPDATED("Changed global variable '%s' value from '%s' to '%s'", RELEASE_EDIT),
    GLOBAL_VARIABLE_PASSWORD_VALUE_UPDATED("Changed global password variable '%s' value", RELEASE_EDIT),
    GLOBAL_VARIABLE_NAME_UPDATED("Changed global variable '%s' name from '%s' to '%s'", RELEASE_EDIT),
    GLOBAL_VARIABLE_LABEL_UPDATED("Changed global variable '%s' label from '%s' to '%s'", RELEASE_EDIT),
    GLOBAL_VARIABLE_DESCRIPTION_UPDATED("Changed global variable '%s' description from '%s' to '%s'", RELEASE_EDIT),
    GLOBAL_VARIABLE_DELETED("Deleted global variable '%s'", RELEASE_EDIT),

    FOLDER_VARIABLE_CREATED("Created folder variable '%s'", RELEASE_EDIT),
    FOLDER_VARIABLE_VALUE_UPDATED("Changed folder variable '%s' value from '%s' to '%s'", RELEASE_EDIT),
    FOLDER_VARIABLE_PASSWORD_VALUE_UPDATED("Changed folder password variable '%s' value", RELEASE_EDIT),
    FOLDER_VARIABLE_DELETED("Deleted folder variable '%s'", RELEASE_EDIT),
    FOLDER_VARIABLE_NAME_UPDATED("Changed folder variable '%s' name from '%s' to '%s'", RELEASE_EDIT),
    FOLDER_VARIABLE_LABEL_UPDATED("Changed folder variable '%s' label from '%s' to '%s'", RELEASE_EDIT),
    FOLDER_VARIABLE_DESCRIPTION_UPDATED("Changed folder variable '%s' description from '%s' to '%s'", RELEASE_EDIT),
    RELEASE_RISK_PROFILE_UPDATED("Changed risk profile from '%s' to '%s'", RELEASE_EDIT),

    TASK_REPORTING_RECORD_CREATED("Created reporting record of type %s", REPORTING_RECORD_EDIT),

    // This is only used during deserialization, as a fallback for obsolete activity names that don't exist anymore as an enum value.
    OTHER("", ActivityCategory.OTHER),

    CUSTOM("%s", ActivityCategory.OTHER);

    private String message;
    private Set<ActivityCategory> categories;

    ReleaseActivity(String message, ActivityCategory... categories) {
        this.message = message;
        Set<ActivityCategory> myCategories = newLinkedHashSet();
        if (categories != null) {
            Collections.addAll(myCategories, categories);
        }
        this.categories = Collections.unmodifiableSet(myCategories);
    }

    public ActivityLogEntry create(Date eventTime, String username, Type targetType, String targetId, Object... parameters) {
        return create(eventTime, name(), username, targetType, targetId, message, parameters);
    }

    public ActivityLogEntry create(XLReleaseEvent event, BaseConfigurationItem target, Object... parameters) {
        return create(event.timestamp(), event.username(), target.getType(), target.getId(), parameters);
    }

    @Override
    public String getName() {
        return name();
    }

    public Set<ActivityCategory> getCategories() {
        return categories;
    }

    @Override
    public String toString() {
        return message;
    }

    private static final Map<String, ReleaseActivity> BY_NAME;

    static {
        ImmutableMap.Builder<String, ReleaseActivity> builder = ImmutableMap.builder();
        for (ReleaseActivity value : ReleaseActivity.values()) {
            builder.put(value.name(), value);
        }
        BY_NAME = builder.build();
    }

    public static ReleaseActivity safeValueOf(String name) {
        ReleaseActivity activity = BY_NAME.get(name);
        return (activity == null ? OTHER : activity);
    }


}
