import React from 'react';
import { DotAutoComplete, DotChip } from '@digital-ai/dot-components';
import { AvatarWithName } from '@xlr-ui/app/features/tasks/components/common/avatar-with-name.component';
import { User } from '@xlr-ui/app/types';
import { mockResizeObserver, mountWithTheme, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { FilterUsername, FilterUsernameProps } from './filter-username.component';
import { RELEASE_LOGO_SVG, SYSTEM_USER } from '../../constants';

describe('FilterUsername', () => {
    const users: User[] = [{ username: 'user1' }, { username: SYSTEM_USER }];
    const onChange = jest.fn();
    let wrapper: ReactWrapper;

    const defaultProps: FilterUsernameProps = {
        inputId: 'input-id',
        onChange,
        placeholder: 'placeholder',
        users,
        value: ['user1'],
    };

    const getAutocomplete = () => wrapper.find(DotAutoComplete);
    const openAutocomplete = () => getAutocomplete().find('input').simulate('mousedown');
    const getAvatar = (id: string) => wrapper.findWhere((node) => node.is(AvatarWithName) && node.prop('id') === id);

    const mount = (props: FilterUsernameProps = defaultProps) => {
        wrapper = mountWithTheme(<FilterUsername {...props} />);
    };

    beforeEach(() => {
        mockResizeObserver();
    });

    it('should render default props', () => {
        mount();
        const autocompleteProps = getAutocomplete().props();
        expect(autocompleteProps.freesolo).toStrictEqual(false);
        expect(autocompleteProps.inputId).toStrictEqual(defaultProps.inputId);
        expect(autocompleteProps.multiple).toStrictEqual(true);
        expect(autocompleteProps.options).toStrictEqual([
            {
                id: SYSTEM_USER,
                title: SYSTEM_USER,
            },
            {
                id: 'user1',
                title: 'user1',
            },
        ]);
        expect(autocompleteProps.placeholder).toStrictEqual(defaultProps.placeholder);
        expect(autocompleteProps.value).toStrictEqual([
            {
                id: 'user1',
                title: 'user1',
            },
        ]);
    });

    it('should render SYSTEM user with logo', () => {
        mount();
        openAutocomplete();
        const systemAvatarProps = getAvatar(SYSTEM_USER).props();
        expect(systemAvatarProps.title).toStrictEqual(SYSTEM_USER);
        expect(systemAvatarProps.avatarImgSrc).toStrictEqual(RELEASE_LOGO_SVG);
    });

    it('should render selected options as chips', () => {
        mount();
        const chip = wrapper.find(DotChip);
        expect(chip).toHaveLength(1);
        expect(chip.prop('children')).toStrictEqual('user1');
        chip.invoke('onDelete')();
        expect(onChange).toHaveBeenCalledWith([]);
    });

    it('should handle onChange', () => {
        mount();
        wrapper.find(DotAutoComplete).prop('onChange')({}, [{ id: SYSTEM_USER }]);
        expect(onChange).toHaveBeenCalledWith([SYSTEM_USER]);
    });
});
