package com.xebialabs.xlrelease.activity

import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.xlrelease.activity.ReleaseActivityDateFormatter.formatDate
import com.xebialabs.xlrelease.domain.ReleaseActivity._
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.{Task, TaskDefinition}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.ActivityLogRepository
import com.xebialabs.xlrelease.repository.IdMatchers.{PhaseId, TaskId}
import com.xebialabs.xlrelease.service.{PhaseService, TaskService}
import com.xebialabs.xlrelease.user.User
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util.Date
import scala.jdk.CollectionConverters._

@Component
@EventListener
class TaskActivityLoggingEventHandler @Autowired()(val eventBus: XLReleaseEventBus,
                                                   val activityLogRepository: ActivityLogRepository,
                                                   taskFieldsComparatorService: TaskFieldsComparatorService,
                                                   phaseService: PhaseService,
                                                   taskService: TaskService)
  extends ActivityLogger[TaskEvent] {

  @Subscribe
  def onEvent(event: TaskEvent): Unit = log(event)


  def logEntries: PartialFunction[XLReleaseEvent, LoggingParams] = {
    onTaskCUDEvent orElse onTaskExecutionEvent
  }

  private val onTaskCUDEvent: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev@TaskCreatedEvent(task) =>
      LoggingParams(releaseIdFrom(task), None, TASK_CREATED.create(ev, task, task.getTitle, task.getTaskType))

    case ev: TaskUpdatedEvent =>
      if (ev.original.getTaskType != ev.updated.getTaskType) {
        onTaskTypeUpdated(ev.timestamp, ev.username, ev.original, ev.updated)
      } else {
        onTaskPropertiesUpdated(ev.timestamp, ev.original, ev.updated, ev.username)
      }

    case ev@TaskDeletedEvent(task) =>
      LoggingParams(releaseIdFrom(task), None, TASK_DELETED.create(ev, task, task.getTitle))

    case ev@TaskCopiedEvent(taskCopy) =>
      LoggingParams(releaseIdFrom(taskCopy), None, TASK_COPIED.create(ev, taskCopy, taskCopy.getTitle))

    case ev@TaskMovedEvent(task, _, _, _, originContainerId, targetContainerId) =>
      if (originContainerId == targetContainerId) {
        LoggingParams(releaseIdFrom(task), None, TASK_MOVED_WITHIN_CONTAINER.create(ev, task,
          task.getTitle, getTaskContainerTitle(originContainerId))
        )
      } else {
        LoggingParams(releaseIdFrom(task), None,
          TASK_MOVED_BETWEEN_CONTAINERS.create(ev, task,
            task.getTitle, getTaskContainerTitle(originContainerId), getTaskContainerTitle(targetContainerId))
        )
      }

    case ev@TasksLockedEvent(releaseId, tasks) =>
      val activities = tasks.asScala.map { task =>
        TASK_LOCKED.create(ev, task, task.getTitle)
      }.asJava
      LoggingParams(Option(releaseId), None, activities)
    case ev@TasksUnlockedEvent(releaseId, tasks) =>
      val activities = tasks.asScala.map { task =>
        TASK_UNLOCKED.create(ev, task, task.getTitle)
      }.asJava
      LoggingParams(Option(releaseId), None, activities)
  }

  private val onTaskExecutionEvent: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev@TaskStartedEvent(task) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_STARTED.create(ev, task, task.getTitle))

    case ev@TaskRetriedEvent(task) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_RESTARTED.create(ev, task, task.getTitle))

    case ev@TaskCompletedEvent(task, false) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_COMPLETED.create(ev, task, task.getTitle))

    case ev@TaskCompletedEvent(task, true) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_COMPLETED_IN_ADVANCE.create(ev, task, task.getTitle))

    case ev@TaskSkippedEvent(task, false) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_SKIPPED.create(ev, task, task.getTitle))

    case ev@TaskSkippedEvent(task, true) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_SKIPPED_IN_ADVANCE.create(ev, task, task.getTitle))

    case ev@TaskFailedEvent(task, reason) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_FAILED.create(ev, task, task.getTitle, reason))

    case ev@TaskReopenedEvent(task) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_REOPENED.create(ev, task, task.getTitle))

    case ev@TaskRecoveryStartedEvent(task) =>
      LoggingParams(releaseIdFrom(task.getId), Some(User.SYSTEM.getName),
        TASK_RECOVERY_STARTED.create(ev, task, task.getTitle, task.getTaskRecoverOp,
          specifiedString(task.isTaskFailureHandlerEnabled && Strings.isNotBlank(task.getFailureHandler))))

    case ev@TaskRecoveredEvent(task) =>
      LoggingParams(releaseIdFrom(task.getId), Some(User.SYSTEM.getName),
        TASK_RECOVERED.create(ev, task, task.getTitle, task.getTaskRecoverOp,
          specifiedString(task.isTaskFailureHandlerEnabled && Strings.isNotBlank(task.getFailureHandler))))

    case ev@TaskAbortScriptStartedEvent(task) =>
      LoggingParams(releaseIdFrom(task.getId), None,
        TASK_ABORT_SCRIPT_STARTED.create(ev, task, task.getTitle, specifiedString(task.hasAbortScript)))

    case ev@TaskAbortScriptCompletedEvent(task) =>
      LoggingParams(releaseIdFrom(task.getId), None,
        TASK_ABORT_SCRIPT_COMPLETED.create(ev, task, task.getTitle, specifiedString(task.hasAbortScript)))

    case ev@TaskGroupFailingEvent(task) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_FAILING.create(ev, task, task.getTitle))

    case ev@TaskWaitingForInputEvent(task, unboundVariables) =>
      LoggingParams(releaseIdFrom(task.getId), None, TASK_WAITING_FOR_INPUT.create(ev, task,
        task.getTitle, unboundVariables.asScala.mkString(", "))
      )

    case ev@TaskDelayedEvent(task) =>
      if (task.isPostponedDueToBlackout) {
        LoggingParams(releaseIdFrom(task.getId), None, TASK_DELAYED_DUE_TO_BLACKOUT.create(ev, task,
          task.getTitle, formatDate(task.getScheduledStartDate))
        )
      } else {
        LoggingParams(releaseIdFrom(task.getId), None, TASK_DELAYED.create(ev, task,
          task.getTitle, formatDate(task.getScheduledStartDate))
        )
      }
  }

  private def specifiedString(condition: Boolean) = if (condition) "specified" else "not specified"

  private def onTaskTypeUpdated(timestamp: Date, username: String, original: Task, updated: Task): LoggingParams = {
    val oldLabel = new TaskDefinition(original.getTaskType, false).getTypeName
    val newLabel = new TaskDefinition(updated.getTaskType, false).getTypeName
    LoggingParams(releaseIdFrom(updated.getId), None,
      TASK_TYPE_CHANGED.create(timestamp, username, original.getType, original.getId, updated.getTitle, oldLabel, newLabel)
    )
  }

  private def onTaskPropertiesUpdated(timestamp: Date, original: Task, updated: Task, username: String): LoggingParams = {
    val logEntries = taskFieldsComparatorService.getLogs(timestamp, username, original, updated)
    LoggingParams(releaseIdFrom(updated.getId), Option(username), logEntries)
  }

  private def getTaskContainerTitle(taskContainerId: String): String = taskContainerId match {
    case PhaseId(id) => phaseService.getTitle(id)
    case TaskId(id) => taskService.getTitle(id)
    case _ => ""
  }
}
