package com.xebialabs.xlrelease.activity

import com.xebialabs.xlrelease.activity.VariablesDiff.getDiffLogEntries
import com.xebialabs.xlrelease.domain.ReleaseActivity._
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.ActivityLogRepository
import com.xebialabs.xlrelease.variable.VariableHelper.withVariableSyntax
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
@EventListener
class ReleaseVariableLoggingEventHandler @Autowired()(val eventBus: XLReleaseEventBus, val activityLogRepository: ActivityLogRepository)
  extends ActivityLogger[ReleaseVariableEvent] {

  @Subscribe
  def onEvent(event: ReleaseVariableEvent): Unit = log(event)

  override def logEntries: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev@ReleaseVariableCreatedEvent(variable) =>
      LoggingParams(releaseIdFrom(variable.getId), None,
        RELEASE_VARIABLE_CREATED.create(ev, variable, withVariableSyntax(variable.getKey)))

    case ev@ReleaseVariableDeletedEvent(variable) =>
      LoggingParams(releaseIdFrom(variable.getId), None,
        RELEASE_VARIABLE_DELETED.create(ev, variable, withVariableSyntax(variable.getKey)))

    case ev@ReleaseVariableUpdatedEvent(original, updated) =>
      val logs = getDiffLogEntries(ev.timestamp, ev.username, original, updated)
      LoggingParams(releaseIdFrom(original.getId), None, logs)

    case ev@ReleaseVariableReplacedEvent(variable, replacement) =>
      LoggingParams(releaseIdFrom(variable.getId), None,
        RELEASE_VARIABLE_REPLACED.create(ev, variable, withVariableSyntax(variable.getKey), replacement))

    case ev@ReleaseVariablesUpdatedEvent(oldVariables, newVariables) =>
      val logs = getDiffLogEntries(ev.timestamp, ev.username, oldVariables, newVariables)
      val maybeReleaseId = newVariables.asScala.headOption.orElse(oldVariables.asScala.headOption)
      val releaseId = maybeReleaseId.flatMap(releaseIdFrom(_)).orElse(None)
      LoggingParams(releaseId, None, logs)
  }


}
