import React from 'react';
import { DotButton, DotChip } from '@digital-ai/dot-components';
import { ActivityLogsFiltersProps, ActivityLogsFilters } from './activity-logs-filters.component';
import { DEFAULT_FILTER_SETTINGS, RELEASE_ACTIVITY_LOG_FILTERS } from '../../constants';
import { mountWithTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';

describe('ActivityLogsFilters', () => {
    const onClearAll = jest.fn();
    const onFilterChange = jest.fn();
    const categories = RELEASE_ACTIVITY_LOG_FILTERS;
    const users = [{ username: 'user1' }, { username: 'user2' }];
    let wrapper: ReactWrapper;

    const getTitle = () => wrapper.find('.filter-group-chips h6');
    const getChips = () => wrapper.find(DotChip);

    const defaultProps: ActivityLogsFiltersProps = {
        categories,
        filters: DEFAULT_FILTER_SETTINGS,
        numberOfFiltersApplied: 1,
        onClearAll,
        onFilterChange,
        dateFormat: 'YYYY-MM-DD',
        timeFormat: 'hh:mm:ss A',
        users,
    };

    const mount = (props: ActivityLogsFiltersProps = defaultProps) => {
        wrapper = mountWithTheme(<ActivityLogsFilters {...props} />);
    };

    it('should render activity types', () => {
        mount({ ...defaultProps, filters: { ...DEFAULT_FILTER_SETTINGS, activityTypes: ['1', '2'] } });
        expect(getTitle()).toHaveText('Activity:');
        const chips = getChips();
        expect(chips).toHaveLength(2);
        expect(chips.at(0)).toHaveText('1');
        chips.at(0).invoke('onDelete')?.();
        expect(onFilterChange).toHaveBeenCalledWith({ ...DEFAULT_FILTER_SETTINGS, activityTypes: ['2'] });
    });

    it('should render usernames', () => {
        mount({ ...defaultProps, filters: { ...DEFAULT_FILTER_SETTINGS, usernames: ['user1'] } });
        expect(getTitle()).toHaveText('Performed by:');
        const chips = getChips();
        expect(chips).toHaveLength(1);
        expect(chips.at(0)).toHaveText('uuser1'); // initial + username
        chips.at(0).invoke('onDelete')?.();
        expect(onFilterChange).toHaveBeenCalledWith({ ...DEFAULT_FILTER_SETTINGS, usernames: [] });
    });

    it('should render categories', () => {
        mount({ ...defaultProps, filters: { ...DEFAULT_FILTER_SETTINGS, lifecycle: true, releaseEdit: true } });
        expect(getTitle()).toHaveText('Categories:');
        const chips = getChips();
        expect(chips).toHaveLength(2);
        expect(chips.at(0)).toHaveText('Release life cycle');
        chips.at(0).invoke('onDelete')?.();
        expect(onFilterChange).toHaveBeenCalledWith({ ...DEFAULT_FILTER_SETTINGS, lifecycle: false, releaseEdit: true });
    });

    it('should render from date', () => {
        mount({ ...defaultProps, filters: { ...DEFAULT_FILTER_SETTINGS, from: 1 } });
        expect(getTitle()).toHaveText('From:');
        const chips = getChips();
        expect(chips).toHaveLength(1);
        expect(chips.at(0)).toHaveText('1970-01-01 - 01:00:00 AM');
        chips.at(0).invoke('onDelete')?.();
        expect(onFilterChange).toHaveBeenCalledWith({ ...DEFAULT_FILTER_SETTINGS, from: null });
    });

    it('should render from date', () => {
        mount({ ...defaultProps, filters: { ...DEFAULT_FILTER_SETTINGS, to: 1 } });
        expect(getTitle()).toHaveText('To:');
        const chips = getChips();
        expect(chips).toHaveLength(1);
        expect(chips.at(0)).toHaveText('1970-01-01 - 01:00:00 AM');
        chips.at(0).invoke('onDelete')?.();
        expect(onFilterChange).toHaveBeenCalledWith({ ...DEFAULT_FILTER_SETTINGS, to: null });
    });

    it('should handle clear all button', () => {
        mount();
        wrapper.find(DotButton).invoke('onClick')?.('' as never);
        expect(onClearAll).toHaveBeenCalled();
    });
});
