import React from 'react';
import { DotBadge, DotIconButton, DotInputText, DotSwitch, DotTypography } from '@digital-ai/dot-components';
import { act, mockResizeObserver, mountWithStoreAndTheme, mountWithTheme, ReactWrapper } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ActivityLog, ActivityLogsContainerEnum, ActivityLogsFilterSettings } from '../types';
import { ActivityLogs, ActivityLogsContent, ActivityLogsProps } from './activity-logs.component';
import { getActivityLogsCategories } from '../helper';
import { ActivityLogsFilterDrawer } from './activity-logs-filter-drawer.component';
import { mockReleaseActivityLogs, mockTriggerActivityLogs } from '../__mocks__/activity-logs.mock';
import { ActivityLogsTable } from './table/activity-logs-table.component';
import { mockIntersectionObserver } from '../../../../../../../../../core/xlr-ui/app/features/release-stepper/__mocks__';
import { activityLogs as activityLogsSlice } from '../ducks/activity-logs.reducer';
import { DEFAULT_FILTER_SETTINGS } from '../constants';
import { ActivityLogsEventDialog } from './table/activity-logs-event-dialog.component';
import { ActionToolbarPortalContainer } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/action-toolbar/components/action-toolbar.portal';
import { releaseMock } from '../../../../../../../../../core/xlr-ui/app/features/tasks/__mocks__/release.mock';

const { reset, setIsDrawerOpened, setHighlightImportant, loadFilteredAndPagedLogs, closeEventDialog, loadEventDialog } = activityLogsSlice.actions;

describe('ActivityLogs component', () => {
    it('should render ActivityLogsContent component', () => {
        const wrapper = mountWithTheme(
            <ActivityLogs containerEnum={ActivityLogsContainerEnum.RELEASE} containerId="1" filterSettings={DEFAULT_FILTER_SETTINGS} release={releaseMock} />,
        );
        const activityLogsContentProps = wrapper.find(ActivityLogsContent).props();
        expect(activityLogsContentProps.containerEnum).toBe(ActivityLogsContainerEnum.RELEASE);
        expect(activityLogsContentProps.containerId).toBe('1');
        expect(activityLogsContentProps.release).toBe(releaseMock);
        expect(activityLogsContentProps.filterSettings).toBe(DEFAULT_FILTER_SETTINGS);
        const portalProps = wrapper.find(ActionToolbarPortalContainer).props();
        expect(portalProps.helpLink).toBe('concept/release-activity-logs.html');
    });
});

describe('ActivityLogsContent component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const observe = jest.fn();
    const unobserve = jest.fn();
    const stopPropagation = jest.fn();
    const dateFormat = 'y-MM-dd';
    const timeFormat = 'HH:mm';
    const activityLogsContainerEnum = ActivityLogsContainerEnum.RELEASE;
    const logs: ActivityLog[] = mockReleaseActivityLogs;
    const filterSettings: ActivityLogsFilterSettings = { ...DEFAULT_FILTER_SETTINGS, folderId: 'Folder1' };

    const defaultProps: ActivityLogsProps = {
        containerEnum: activityLogsContainerEnum,
        containerId: '1',
        filterSettings,
        release: releaseMock,
    };

    const defaultState = {
        activityLogs: {
            logs,
            logsFilter: {
                containerId: defaultProps.containerId,
                filterSettings: {
                    ...defaultProps.filterSettings,
                    filter: 'my text filter',
                    activityTypes: ['ATTACHMENT_ADDED_ON_TASK', 'COMMENT_ADDED'],
                },
            },
            activityTypes: ['ATTACHMENT_ADDED_ON_TASK', 'COMMENT_ADDED'],
            users: [
                {
                    username: 'admin',
                    fullName: 'Release Administrator',
                    email: 'releaseadmin@dai.com',
                },
            ],
            eventDetails: 'my trigger event details',
            isEventDialogOpened: true,
            isDrawerOpened: false,
            isLastPage: false,
            isLoading: false,
        },
        profile: {
            dateFormat,
            timeFormat,
        },
    };

    const getTitleTypography = () => wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'activity-logs-title-typography');
    const getDotSwitch = () => wrapper.find(DotSwitch);
    const getFilterInput = () => wrapper.findWhere((node) => node.is(DotInputText) && node.props().id === 'activity-logs-filter');
    const getDotBadge = () => wrapper.find(DotBadge);
    const getBadgeIconButton = () => getDotBadge().find(DotIconButton);
    const getActivityLogsTable = () => wrapper.find(ActivityLogsTable);
    const getActivityLogsFilterDrawer = () => wrapper.find(ActivityLogsFilterDrawer);
    const getActivityLogsEventDialog = () => wrapper.find(ActivityLogsEventDialog);

    const mount = async (props: ActivityLogsProps = defaultProps, state = defaultState) => {
        await act(async () => {
            wrapper = mountWithStoreAndTheme(<ActivityLogsContent {...props} />, dispatch, state);
        });
    };

    beforeEach(() => {
        mockIntersectionObserver(observe, unobserve);
        mockResizeObserver();
    });

    afterEach(() => {
        jest.clearAllMocks();
    });

    it('should render properly', async () => {
        await mount();
        const titleTypography = getTitleTypography();
        expect(titleTypography).toExist();
        const titleTypographyProps = titleTypography.props();
        expect(titleTypographyProps.variant).toBe('h1');
        expect(titleTypographyProps.children).toBe('Activity logs');

        const dotSwitch = getDotSwitch();
        expect(dotSwitch).toExist();
        const dotSwitchProps = dotSwitch.props();
        expect(dotSwitchProps.checked).toBe(true);
        expect(dotSwitchProps.label).toBe('Highlight important');
        expect(dotSwitchProps.onChange).toStrictEqual(expect.any(Function));

        const filterInput = getFilterInput();
        expect(filterInput).toExist();
        const filterInputProps = filterInput.props();
        expect(filterInputProps.name).toBe('activity-logs-filter');
        expect(filterInputProps.placeholder).toBe('Filter by activity message, ID or user');

        const dotBadge = getDotBadge();
        expect(dotBadge).toExist();
        const dotBadgeProps = dotBadge.props();
        expect(dotBadgeProps.badgeContent).toBe(1);
        expect(dotBadgeProps.badgeColor).toBe('#d61f21');
        expect(dotBadgeProps.overlap).toBe('circular');
        expect(dotBadgeProps.variant).toBe('standard');

        const badgeIconButton = getBadgeIconButton();
        expect(badgeIconButton).toExist();
        const badgeIconButtonProps = badgeIconButton.props();
        expect(badgeIconButtonProps.className).toBe('filter-icon-btn');
        expect(badgeIconButtonProps.iconId).toBe('filter');
        expect(badgeIconButtonProps.onClick).toStrictEqual(expect.any(Function));

        const activityLogsTable = getActivityLogsTable();
        expect(activityLogsTable).toExist();
        const activityLogsTableProps = activityLogsTable.props();
        expect(activityLogsTableProps.dateOrder).toBe('desc');
        expect(activityLogsTableProps.containerEnum).toBe(activityLogsContainerEnum);
        expect(activityLogsTableProps.isLastPage).toBe(false);
        expect(activityLogsTableProps.isImportantHighlighted).toBe(true);
        expect(activityLogsTableProps.isLoading).toBe(false);
        expect(activityLogsTableProps.validTargetIds).toStrictEqual(['Release1-Phase1-Task1']);
        expect(activityLogsTableProps.logs).toStrictEqual(logs);
        expect(activityLogsTableProps.onIntersect).toStrictEqual(expect.any(Function));
        expect(activityLogsTableProps.onUpdateTableData).toStrictEqual(expect.any(Function));
        expect(activityLogsTableProps.dateFormat).toBe('YYYY-MM-DD');
        expect(activityLogsTableProps.timeFormat).toBe(timeFormat);

        const activityLogsFilterDrawer = getActivityLogsFilterDrawer();
        expect(activityLogsFilterDrawer).toExist();
        const activityLogsFilterDrawerProps = activityLogsFilterDrawer.props();
        expect(activityLogsFilterDrawerProps.activityTypes).toStrictEqual([
            {
                id: 'ATTACHMENT_ADDED_ON_TASK',
                title: 'Attachment added on task',
            },
            { id: 'COMMENT_ADDED', title: 'Comment added' },
        ]);
        expect(activityLogsFilterDrawerProps.categories).toStrictEqual(getActivityLogsCategories(defaultProps.containerEnum));
        expect(activityLogsFilterDrawerProps.filter).toStrictEqual({
            activityTypes: ['ATTACHMENT_ADDED_ON_TASK', 'COMMENT_ADDED'],
            comments: false,
            dateAsc: false,
            deliveryEdit: false,
            execution: false,
            filter: 'my text filter',
            folderId: 'Folder1',
            from: null,
            isImportantHighlighted: true,
            lifecycle: false,
            other: false,
            reassign: false,
            releaseEdit: false,
            reportingRecordEdit: false,
            security: false,
            taskEdit: false,
            to: null,
            triggerEdit: false,
            usernames: [],
        });
        expect(activityLogsFilterDrawerProps.isDrawerOpened).toBe(false);
        expect(activityLogsFilterDrawerProps.numberOfFiltersApplied).toBe(1);
        expect(activityLogsFilterDrawerProps.onClearAll).toStrictEqual(expect.any(Function));
        expect(activityLogsFilterDrawerProps.onDrawerClose).toStrictEqual(expect.any(Function));
        expect(activityLogsFilterDrawerProps.onFilterChange).toStrictEqual(expect.any(Function));
        expect(activityLogsFilterDrawerProps.users).toStrictEqual([
            {
                email: 'releaseadmin@dai.com',
                fullName: 'Release Administrator',
                username: 'admin',
            },
        ]);

        const dialogProps = getActivityLogsEventDialog().props();
        expect(dialogProps.content).toBe('my trigger event details');
        expect(dialogProps.onClose).toStrictEqual(expect.any(Function));
    });

    it('should NOT render Highlight important switch for trigger container', async () => {
        await mount(
            { ...defaultProps, containerEnum: ActivityLogsContainerEnum.TRIGGER },
            {
                ...defaultState,
                activityLogs: {
                    logs: mockTriggerActivityLogs,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        filterSettings: defaultProps.filterSettings,
                    },
                    activityTypes: [],
                    users: [],
                    eventDetails: 'my trigger event details',
                    isEventDialogOpened: false,
                    isDrawerOpened: false,
                    isLastPage: false,
                    isLoading: false,
                },
            },
        );
        const dotSwitch = getDotSwitch();
        expect(dotSwitch).not.toExist();
    });

    describe('dispatch actions', () => {
        it('should dispatch reset on unmount', async () => {
            await mount();
            wrapper.unmount();
            expect(dispatch).toHaveBeenCalledWith(reset());
        });

        it('should dispatch correct action when drawer is opened', async () => {
            await mount();
            const badgeIconButton = getBadgeIconButton();
            badgeIconButton.invoke('onClick')({
                currentTarget: document.createElement('div'),
                stopPropagation,
            } as never);
            expect(dispatch).toHaveBeenCalledWith(setIsDrawerOpened(true));
        });

        it('should dispatch correct action when cleaning all filters', async () => {
            await mount();
            const activityLogsFilterDrawer = getActivityLogsFilterDrawer();
            activityLogsFilterDrawer.invoke('onClearAll')();
            expect(dispatch).toHaveBeenCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        filterSettings: { ...DEFAULT_FILTER_SETTINGS, filter: 'my text filter', folderId: filterSettings.folderId },
                        containerEnum: defaultProps.containerEnum,
                    },
                }),
            );
        });

        it('should dispatch correct action when changing filters', async () => {
            await mount();
            const activityLogsFilterDrawer = getActivityLogsFilterDrawer();
            activityLogsFilterDrawer.invoke('onFilterChange')({ comments: true } as never);
            expect(dispatch).toHaveBeenCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        containerEnum: defaultProps.containerEnum,
                        filterSettings: { comments: true } as never,
                    },
                }),
            );
        });

        it('should dispatch correct action when onChange handler has been executed on the filter input', async () => {
            await mount();
            const filterInput = getFilterInput();
            filterInput.invoke('onChange')({ target: { value: 'new filter' } } as never);
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        containerEnum: defaultProps.containerEnum,
                        filterSettings: expect.objectContaining({
                            filter: 'new filter',
                        }),
                    },
                }),
            );
        });

        it('should dispatch correct action when onClear handler has been executed on the filter input', async () => {
            await mount();
            const filterInput = getFilterInput();
            filterInput.invoke('onClear')();
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        containerEnum: defaultProps.containerEnum,
                        filterSettings: expect.objectContaining({
                            filter: '',
                        }),
                    },
                }),
            );
        });

        it('should dispatch correct action when onUpdateTableData is executed', async () => {
            await mount();
            const isDateAsc = true;
            const activityLogsTable = getActivityLogsTable();
            activityLogsTable.invoke('onUpdateTableData')(isDateAsc);
            expect(dispatch).toHaveBeenLastCalledWith(
                loadFilteredAndPagedLogs({
                    page: 0,
                    logsFilter: {
                        containerId: defaultProps.containerId,
                        containerEnum: defaultProps.containerEnum,
                        filterSettings: expect.objectContaining({
                            dateAsc: isDateAsc,
                        }),
                    },
                }),
            );
        });

        it('should execute correct action when DotSwitch has been checked', async () => {
            await mount();
            const dotSwitch = getDotSwitch();
            dotSwitch.invoke('onChange')();
            expect(dispatch).toHaveBeenLastCalledWith(setHighlightImportant(false));
        });

        it('should dispatch correct actions for the event dialog', async () => {
            await mount();
            const activityLogsTable = getActivityLogsTable();
            activityLogsTable.invoke('onDetailsClick')('my data Id');
            expect(dispatch).toHaveBeenCalledWith(loadEventDialog('my data Id'));

            const dialog = getActivityLogsEventDialog();
            dialog.invoke('onClose')();
            expect(dispatch).toHaveBeenCalledWith(closeEventDialog());
        });
    });
});
