import { all, call, put, select, takeLatest } from 'redux-saga/effects';
import { PayloadAction } from '@reduxjs/toolkit';
import { SagaIterator } from 'redux-saga';
import { httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { activityLogs, ActivityLogsState, getActivityLogsState, LoadFilteredAndPagedLogsAction, LogsFilter } from './activity-logs.reducer';
import { ActivityLog, ActivityLogsClientSettings, ActivityLogsFilterSettings } from '../types';
import { LOG_USER, LOGS_ITEMS_PER_PAGE, SYSTEM_USER } from '../constants';
import { checkIfLastPage, getFilterSettingsForLoadPayload } from '../helper';
import { FilterQueryParams, User } from '../../../../../../../../../core/xlr-ui/app/types';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';

const {
    init,
    loadEventDialog,
    loadFilteredAndPagedLogs,
    setActivityTypes,
    setEventDetails,
    setEventDialogOpened,
    setHighlightImportant,
    setIsLastPage,
    setIsLoading,
    setLogs,
    setLogsFilter,
    setPage,
    setUsers,
} = activityLogs.actions;

export function* initSaga(action: PayloadAction<LogsFilter>): SagaIterator {
    try {
        const page = 0;
        const logsFilter = action.payload;
        yield put(setIsLoading(true));
        yield put(setPage(page));
        const [{ data: activityTypes }, { data: users }] = yield all([
            call(httpGET, `activity/${logsFilter.containerId}/logs/activity-types`),
            call(httpGET, `activity/${logsFilter.containerId}/logs/users`),
        ]);
        const usersWithoutTaskComment = replaceLogUserWithSystemUser(users);
        yield put(setActivityTypes(activityTypes));
        yield put(setUsers(usersWithoutTaskComment));

        const cleanedLogFilters = cleanLogFilters(logsFilter, activityTypes, usersWithoutTaskComment);
        yield put(setLogsFilter(cleanedLogFilters));
        const logs = yield call(loadLogs, page, cleanedLogFilters);

        const isLastPage = checkIfLastPage(logs, LOGS_ITEMS_PER_PAGE);
        yield put(setIsLastPage(isLastPage));
        yield put(setLogs(logs));
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* loadFilteredAndPagedLogsSaga(action: PayloadAction<LoadFilteredAndPagedLogsAction>) {
    try {
        const { logsFilter, page } = action.payload;
        yield put(setIsLoading(true));
        yield put(setPage(page));
        const { logs: oldLogs }: ActivityLogsState = yield select(getActivityLogsState);
        yield put(setLogsFilter(logsFilter));
        const logs: ActivityLog[] = yield call(loadLogs, page, logsFilter);
        let newLogs: ActivityLog[] = oldLogs;

        if (page === 0) {
            newLogs = logs;
        } else if (logs.length > 0) {
            newLogs = oldLogs.concat(logs);
        }

        const isLastPage = checkIfLastPage(logs, LOGS_ITEMS_PER_PAGE);
        yield put(setIsLastPage(isLastPage));
        yield put(setLogs(newLogs));
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* loadEventDialogSaga(action: PayloadAction<string>) {
    yield put(setEventDialogOpened(true));
    const { data } = yield call(httpGET, `activity/data/${action.payload}`);
    if (data.content) {
        try {
            // eslint-disable-next-line angular/json-functions
            data.content = JSON.parse(data.content);
        } catch (_e) {
            // ignore the error if the content is not JSON
        }
    }
    // eslint-disable-next-line angular/json-functions
    yield put(setEventDetails(JSON.stringify(data, null, 2)));
}

export function* loadLogs(page: number, logsFilter: LogsFilter) {
    const logsFilterWithTaskComment = getFiltersWithLogUser(logsFilter);
    const { data: logs } = yield call(
        httpPOST,
        `activity/${logsFilter.containerId}/logs/search?page=${page}&resultsPerPage=${LOGS_ITEMS_PER_PAGE}`,
        getFilterSettingsForLoadPayload(logsFilterWithTaskComment.filterSettings),
    );
    // we don't store the logsFilterWithTaskComment in the url since it might contain the task comment user
    yield call(updateClientSettingsAndUrl, logsFilter.containerId, logsFilter.filterSettings);
    return logs;
}

export function* setHighlightImportantSaga(action: PayloadAction<boolean>) {
    const { logsFilter }: ActivityLogsState = yield select(getActivityLogsState);
    const newFilterSettings = { ...logsFilter.filterSettings, isImportantHighlighted: action.payload };
    yield put(setLogsFilter({ ...logsFilter, filterSettings: newFilterSettings }));
    yield call(updateClientSettingsAndUrl, logsFilter.containerId, newFilterSettings);
}

export function cleanLogFilters(logsFilter: LogsFilter, activityTypes: string[], users: User[]) {
    // when manually typing to url some value that is not present in filters drawer, then remove those invalid filters
    return {
        ...logsFilter,
        filterSettings: {
            ...logsFilter.filterSettings,
            usernames: logsFilter.filterSettings.usernames.filter((username) => users.some((u) => u.username === username)),
            activityTypes: logsFilter.filterSettings.activityTypes.filter((activityType) => activityTypes.includes(activityType)),
        },
    };
}

export function replaceLogUserWithSystemUser(users: User[]): User[] {
    const hasLogUser = users.some((user) => user.username === LOG_USER);
    if (hasLogUser) {
        const hasSystemUser = users.some((user) => user.username === SYSTEM_USER);
        if (hasSystemUser) {
            // remove log user if system user is present, since we are only showing system user
            return users.filter((user) => user.username !== LOG_USER);
        } else {
            // replace log user with system user
            return users.map((user) => (user.username === LOG_USER ? { username: SYSTEM_USER } : user));
        }
    }
    return users;
}

export function getFiltersWithLogUser(logsFilter: LogsFilter) {
    if (logsFilter.filterSettings.usernames.includes(SYSTEM_USER)) {
        return {
            ...logsFilter,
            filterSettings: {
                ...logsFilter.filterSettings,
                usernames: [...logsFilter.filterSettings.usernames, LOG_USER],
            },
        };
    }
    return logsFilter;
}

export function* updateClientSettingsAndUrl(containerId: string, filterSettings: ActivityLogsFilterSettings) {
    const FiltersQueryParams: FilterQueryParams = yield call(getAngularService, 'FiltersQueryParams');
    const clientSettings: ActivityLogsClientSettings = yield call(getAngularService, 'ClientSettings');
    clientSettings.setLogsFilters(containerId, filterSettings);
    FiltersQueryParams.update(filterSettings);
}

export default function* activityLogsSaga() {
    yield all([
        takeLatest(init, initSaga),
        takeLatest(loadFilteredAndPagedLogs, loadFilteredAndPagedLogsSaga),
        takeLatest(setHighlightImportant, setHighlightImportantSaga),
        takeLatest(loadEventDialog, loadEventDialogSaga),
    ]);
}
