package com.xebialabs.xlrelease.activity

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.domain.{Release, Trigger}
import com.xebialabs.xlrelease.domain.ReleaseActivity._
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.{ActivityLogRepository, TriggerRepository}
import com.xebialabs.xlrelease.repository.Ids.getParentId
import com.xebialabs.xlrelease.service.{ReleaseService, UserInfoResolver}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@EventListener
class ReleaseActivityLoggingEventHandler @Autowired()(val eventBus: XLReleaseEventBus,
                                                      val activityLogRepository: ActivityLogRepository,
                                                      releaseService: ReleaseService,
                                                      userInfoResolver: UserInfoResolver,
                                                     triggerRepository: TriggerRepository)
  extends ActivityLogger[ReleaseEvent] {

  @Subscribe
  def onEvent(event: ReleaseEvent): Unit = log(event)

  override def logEntries: PartialFunction[XLReleaseEvent, LoggingParams] = {
    onTemplateCUDEvent orElse onReleaseCUDEvent orElse onReleaseExecutionEvent
  }

  private val onTemplateCUDEvent: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev: ReleaseCreatedEvent if ev.release.isTemplate && ev.source == null =>
      LoggingParams(releaseIdFrom(ev.release), Option(ev.username),
        TEMPLATE_CREATED.create(ev, ev.release)
      )

    case ev: ReleaseCreatedEvent if ev.release.isTemplate && ev.source.isInstanceOf[Imported] =>
      LoggingParams(releaseIdFrom(ev.release), Option(ev.username),
        TEMPLATE_IMPORTED.create(ev, ev.release)
      )

    case ev: ReleaseDuplicatedEvent if ev.releaseDuplicate.isTemplate =>
      LoggingParams(releaseIdFrom(ev.releaseDuplicate), Option(ev.username),
        TEMPLATE_DUPLICATED.create(ev, ev.releaseDuplicate, ev.releaseDuplicate.getTitle)
      )

    case ev: ReleaseMovedEvent if releaseService.isTemplate(ev.toReleaseId) =>
      LoggingParams(Some(ev.toReleaseId), Option(ev.username),
        TEMPLATE_MOVED.create(ev.timestamp, ev.username, Type.valueOf(classOf[Release]), ev.fromReleaseId,
          getParentId(ev.fromReleaseId), getParentId(ev.toReleaseId)
        )
      )

    case ev@ReleaseCreatedEvent(release, CreatedFromAsCode(scmTraceabilityData)) =>
      LoggingParams(releaseIdFrom(ev.release), Option(ev.username),
        TEMPLATE_CREATED_FROM_AS_CODE.create(ev, release,
          ReleaseActivityLoggingEventHandler.getScmTraceabilityDataInfo(scmTraceabilityData)
        )
      )

    case ev: ReleaseUpdatedFromAsCodeEvent =>
      LoggingParams(releaseIdFrom(ev.release), Option(ev.username),
        TEMPLATE_UPDATED_FROM_AS_CODE.create(ev, ev.release,
          ReleaseActivityLoggingEventHandler.getScmTraceabilityDataInfo(ev.scmTraceabilityData)
        )
      )
  }

  private val onReleaseCUDEvent: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev@ReleaseCreatedEvent(release, null) =>
      LoggingParams(Some(release.getId), None,
        RELEASE_CREATED.create(ev, release, release.getTitle)
      )

    case ev@ReleaseCreatedEvent(release, CreatedFromTemplateByTrigger(templateId, triggerId)) =>
      val triggerTitle = triggerRepository.find[Trigger](triggerId).getTitle
      LoggingParams(Some(release.getId), None,
        RELEASE_CREATED_FROM_TRIGGER.create(ev, release, release.getTitle, triggerTitle, releaseService.getTitle(templateId))
      )

    case ev@ReleaseCreatedEvent(release, CreatedFromTemplate(templateId)) =>
      if (release.isWorkflow) {
        LoggingParams(Some(release.getId), None,
          WORKFLOW_EXECUTION_CREATED_FROM_TEMPLATE.create(ev, release, release.getTitle, releaseService.getTitle(templateId))
        )
      } else {
        LoggingParams(Some(release.getId), None,
          RELEASE_CREATED_FROM_TEMPLATE.create(ev, release, release.getTitle, releaseService.getTitle(templateId))
        )
      }

    case ev@ReleaseCreatedEvent(release, CreatedFromCreateReleaseTask(task, templateId)) =>
      LoggingParams(Some(task.getRelease.getId), None,
        RELEASE_CREATED_FROM_CREATE_RELEASE_TASK.create(ev, release,
          release.getTitle, task.getTitle, releaseService.getTitle(templateId)
        )
      )

    case ev@ReleaseCreatedEvent(release, CreatedFromDsl()) =>
      if (release.isWorkflow) {
        if (release.isTemplate) {
          LoggingParams(Some(release.getId), None,
            WORKFLOW_TEMPLATE_CREATED_FROM_DSL.create(ev, release, release.getTitle)
          )
        } else {
          LoggingParams(Some(release.getId), None,
            WORKFLOW_EXECUTION_CREATED_FROM_DSL.create(ev, release, release.getTitle)
          )
        }
      } else {
        LoggingParams(Some(release.getId), None,
          RELEASE_CREATED_FROM_DSL.create(ev, release, release.getTitle)
        )
      }

    case ev: ReleaseUpdatedEvent =>
      val logs = new ReleaseFieldsComparator(ev.timestamp, ev.username, ev.original, ev.updated, userInfoResolver).getLogs
      logs.addAll(VariablesDiff.getDiffLogEntries(ev.timestamp, ev.username, ev.original.getVariables, ev.updated.getVariables))
      LoggingParams(Some(ev.updated.getId), Option(ev.username), logs)

    case ev: ReleaseRestoredEvent =>
      LoggingParams(Some(ev.original.getId), Option(ev.username),
        RELEASE_RESTORED_FROM_REVISION.create(ev, ev.original, ev.original.getTitle, ev.revisionId.toString)
      )
  }

  private val onReleaseExecutionEvent: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev@ReleaseStartedEvent(release, _) =>
      if (release.isWorkflow) {
        LoggingParams(Some(release.getId), None,
          WORKFLOW_EXECUTION_STARTED.create(ev, release)
        )
      } else {
        LoggingParams(Some(release.getId), None,
          RELEASE_STARTED.create(ev, release)
        )
      }

    case ev@ReleaseFailedEvent(release) =>
      if (release.isWorkflow) {
        LoggingParams(Some(release.getId), None,
          WORKFLOW_EXECUTION_FAILED.create(ev, release))
      } else {
        LoggingParams(Some(release.getId), None,
          RELEASE_FAILED.create(ev, release))
      }

    case ev@ReleaseStartedFailingEvent(release) =>
      if (release.isWorkflow) {
        LoggingParams(Some(release.getId), None,
          WORKFLOW_EXECUTION_FAILING.create(ev, release))
      } else {
        LoggingParams(Some(release.getId), None,
          RELEASE_FAILING.create(ev, release))
      }

    case ev@ReleaseRetriedEvent(release) =>
      if (release.isWorkflow) {
        LoggingParams(Some(release.getId), None,
          WORKFLOW_EXECUTION_RESTARTED.create(ev, release))
      } else {
        LoggingParams(Some(release.getId), None,
          RELEASE_RESTARTED.create(ev, release))
      }

    case ev@ReleaseCompletedEvent(release) =>
      if (release.isWorkflow) {
        LoggingParams(Some(release.getId), None,
          WORKFLOW_EXECUTION_COMPLETED.create(ev, release))
      } else {
        LoggingParams(Some(release.getId), None,
          RELEASE_COMPLETED.create(ev, release))
      }

    case ev@ReleaseAbortedEvent(release, _) =>
      if (release.isWorkflow) {
        LoggingParams(Some(release.getId), None, WORKFLOW_EXECUTION_ABORTED.create(ev, release, release.getAbortComment))
      } else {
        LoggingParams(Some(release.getId), None, RELEASE_ABORTED.create(ev, release, release.getAbortComment))
      }

    case ev@ReleaseResumedEvent(release) =>
      if (release.isWorkflow) {
        LoggingParams(Some(release.getId), None,
          WORKFLOW_EXECUTION_RESTARTED.create(ev, release))
      } else {
        LoggingParams(Some(release.getId), None,
          RELEASE_RESTARTED.create(ev, release))
      }

    case ev@ReleaseStartedFromCreateReleaseTaskEvent(task, subRelease) =>
      LoggingParams(Some(task.getRelease.getId), None,
        RELEASE_STARTED_FROM_CREATE_RELEASE_TASK.create(ev, subRelease, subRelease.getTitle, task.getTitle))
  }
}

object ReleaseActivityLoggingEventHandler {

  def getScmTraceabilityDataInfo(scmTraceabilityData: Option[SCMTraceabilityData]): String = {
    scmTraceabilityData match {
      case Some(scmData) => s"via Source Control Management with meta information: \nCommit: ${scmData.getKind}-${scmData.getCommit}" +
        s" \nTimestamp: ${scmData.getDate} \nCommitted by: ${scmData.getAuthor} \nSummary: ${scmData.getMessage} \nSource: ${scmData.getRemote}" +
        s" \nFile name: ${scmData.getFileName} \nCommit link: ${scmData.getRemote}/commit/${scmData.getCommit}"
      case None => ""
    }
  }

}
