package com.xebialabs.xlrelease.activity

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.ReleaseActivity._
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.{ActivityLogRepository, Ids}
import com.xebialabs.xlrelease.service.ReleaseService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@EventListener
class TeamActivityLoggingEventHandler @Autowired()(val eventBus: XLReleaseEventBus,
                                                   val activityLogRepository: ActivityLogRepository,
                                                   releaseService: ReleaseService)
  extends ActivityLogger[TeamEvent] {

  @Subscribe
  def onEvent(event: TeamEvent): Unit = log(event)

  override def logEntries: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev@TeamCreatedEvent(containerId, team) =>
      LoggingParams(Some(containerId), None, TEAM_CREATED.create(ev, team, team.getTeamName))

    case ev@TeamUpdatedEvent(containerId, originalTeam, updatedTeam) =>
      LoggingParams(Some(containerId), None,
        TEAM_UPDATED.create(ev, originalTeam, updatedTeam.getTeamName, updatedTeam.getMembers.toString, updatedTeam.getRoles.toString)
      )

    case ev@TeamDeletedEvent(containerId, team) =>
      LoggingParams(Some(containerId), None, TEAM_DELETED.create(ev, team, team.getTeamName))

    // FIXME: need folderId
    case ev@TeamsMergedEvent(templateId, folderTitle) =>
      LoggingParams(Some(templateId), None,
        FOLDER_TEAM_MERGED.create(ev.timestamp, ev.username, Type.valueOf(classOf[Folder]), Ids.findFolderId(templateId),
          releaseService.getTitle(templateId), folderTitle)
      )

    // FIXME: need folderId
    case ev@TeamsRemovedInTemplateEvent(templateId, folderTitle) =>
      LoggingParams(Some(templateId), None,
        REMOVE_TEMPLATE_TEAMS.create(ev.timestamp, ev.username, Type.valueOf(classOf[Folder]), Ids.findFolderId(templateId),
          releaseService.getTitle(templateId), folderTitle
        )
      )
  }
}
