import { ActivityLog, ActivityLogsContainerEnum } from '../types';
import {
    calculateActivityLogColumnProps,
    checkIfImportantActivity,
    checkIfLastPage,
    countFiltersApplied,
    generateIdFromUsername,
    getActivityLogColumnMessage,
    getActivityLogsCategories,
    getActivityTypeLabel,
    getSelectedCategoryFromActivitySettings,
    getSelectedCategoryFromIds,
    getSystemUserAvatar,
    getTableColumns,
    getTargetTypeLabel,
    getTaskWithOpenedModalHref,
    checkIfMessageIsMarkdown,
    removeAllAfterLastTask,
    getFilterSettingsForLoadPayload,
    getReleaseHref,
    getDeliveryUrlFromPath,
} from './index';
import {
    DELIVERY_ACTIVITY_LOG_CATEGORIES,
    IMPORTANT_ACTIVITY_TYPES,
    RELEASE_ACTIVITY_LOG_FILTERS,
    RELEASE_LOGO_SVG,
    SYSTEM_USER,
    TRIGGER_ACTIVITY_LOG_CATEGORIES,
} from '../constants';
import colors from '../../../../../../../../../core/xlr-ui/app/js/colors';

describe('activity log helpers', () => {
    describe('getActivityLogsCategories', () => {
        it('should return DELIVERY_ACTIVITY_LOG_CATEGORIES when enum is DELIVERY', () => {
            const result = getActivityLogsCategories(ActivityLogsContainerEnum.DELIVERY);
            expect(result).toStrictEqual(DELIVERY_ACTIVITY_LOG_CATEGORIES);
        });

        it('should return RELEASE_ACTIVITY_LOG_FILTERS when enum is RELEASE', () => {
            const result = getActivityLogsCategories(ActivityLogsContainerEnum.RELEASE);
            expect(result).toStrictEqual(RELEASE_ACTIVITY_LOG_FILTERS);
        });

        it('should return TRIGGER_ACTIVITY_LOG_CATEGORIES when enum is TRIGGER', () => {
            const result = getActivityLogsCategories(ActivityLogsContainerEnum.TRIGGER);
            expect(result).toStrictEqual(TRIGGER_ACTIVITY_LOG_CATEGORIES);
        });

        it('should return RELEASE_ACTIVITY_LOG_FILTERS for an unknown enum value (default case)', () => {
            const invalidEnumValue = -1;
            const result = getActivityLogsCategories(invalidEnumValue as ActivityLogsContainerEnum);
            expect(result).toBe(RELEASE_ACTIVITY_LOG_FILTERS);
        });
    });

    describe('getSelectedCategoryFromActivitySettings', () => {
        it('should return keys with boolean `true` values', () => {
            const settings = {
                comments: true,
                dateAsc: false,
                deliveryEdit: true,
                filter: 'someString',
            };
            const result = getSelectedCategoryFromActivitySettings(settings as never);
            expect(result).toEqual(['comments', 'deliveryEdit']);
        });

        it('should return an empty array if no boolean `true` values exist', () => {
            const settings = {
                comments: false,
                dateAsc: false,
                filter: 'someString',
            };
            const result = getSelectedCategoryFromActivitySettings(settings as never);
            expect(result).toEqual([]);
        });
    });

    describe('checkIfLastPage', () => {
        const logs: ActivityLog[] = [
            {
                activityType: 'STAGE_STARTED',
            },
            {
                activityType: 'TRIGGER_ENABLED',
            },
        ] as never;

        test('should return true if logs array is empty', () => {
            const result = checkIfLastPage([], 10);
            expect(result).toBe(true);
        });

        test('should return true if logs length is less than logsPerPage', () => {
            const result = checkIfLastPage(logs, 5);
            expect(result).toBe(true);
        });

        test('should return false if logs length is equal to logsPerPage', () => {
            const result = checkIfLastPage(logs, 2);
            expect(result).toBe(false);
        });

        test('should return false if logs length is greater than logsPerPage', () => {
            const result = checkIfLastPage(logs, 1);
            expect(result).toBe(false);
        });
    });

    describe('getSelectedCategoryFromIds', () => {
        it('should return correct categories for given ids', () => {
            const ids = ['comments', 'deliveryEdit', 'security'];
            const result = getSelectedCategoryFromIds(ids);
            expect(result).toEqual({
                comments: true,
                deliveryEdit: true,
                execution: false,
                lifecycle: false,
                other: false,
                reassign: false,
                releaseEdit: false,
                reportingRecordEdit: false,
                security: true,
                taskEdit: false,
                triggerEdit: false,
            });
        });

        it('should return all false for empty ids', () => {
            const ids: string[] = [];
            const result = getSelectedCategoryFromIds(ids);
            expect(result).toEqual({
                comments: false,
                deliveryEdit: false,
                execution: false,
                lifecycle: false,
                other: false,
                reassign: false,
                releaseEdit: false,
                reportingRecordEdit: false,
                security: false,
                taskEdit: false,
                triggerEdit: false,
            });
        });
    });

    describe('getActivityTypeLabel', () => {
        it('should format activity type correctly', () => {
            expect(getActivityTypeLabel('STAGE_STARTED')).toBe('Stage started');
            expect(getActivityTypeLabel('TRIGGER_ENABLED')).toBe('Trigger enabled');
            expect(getActivityTypeLabel('RELEASE_COMPLETED')).toBe('Release completed');
        });
    });

    describe('countFiltersApplied', () => {
        it('should return 0 when no filters are applied', () => {
            const filter = {
                filter: '',
                dateAsc: true,
                other: false,
                comments: false,
                isImportantHighlighted: false,
                lifecycle: false,
                releaseEdit: false,
                triggerEdit: false,
                deliveryEdit: false,
                taskEdit: false,
                security: false,
                execution: false,
                reassign: false,
                reportingRecordEdit: false,
                activityTypes: [],
                usernames: [],
                folderId: null,
                from: null,
                to: null,
            };
            const result = countFiltersApplied(filter);
            expect(result).toBe(0);
        });

        it('should return the correct number of filters applied', () => {
            const filter = {
                filter: '',
                dateAsc: true,
                other: true,
                comments: false,
                isImportantHighlighted: false,
                lifecycle: true,
                releaseEdit: false,
                triggerEdit: false,
                deliveryEdit: false,
                taskEdit: false,
                security: false,
                execution: false,
                reassign: false,
                reportingRecordEdit: false,
                activityTypes: ['type1'],
                usernames: ['user1'],
                folderId: null,
                from: 5555,
                to: null,
            };
            const result = countFiltersApplied(filter);
            expect(result).toBe(4);
        });

        it('should count all boolean filters and array filters correctly', () => {
            const filter = {
                filter: '',
                dateAsc: true,
                other: true,
                comments: true,
                lifecycle: true,
                isImportantHighlighted: false,
                releaseEdit: true,
                triggerEdit: true,
                deliveryEdit: true,
                taskEdit: true,
                security: true,
                execution: true,
                reassign: true,
                reportingRecordEdit: true,
                activityTypes: ['type1', 'type2'],
                usernames: ['user1', 'user2'],
                folderId: 'Applications/Folder1',
                from: 7777,
                to: 8888,
            };
            const result = countFiltersApplied(filter);
            expect(result).toBe(5);
        });
    });

    describe('getTableColumns', () => {
        beforeEach(() => {
            Intl.DateTimeFormat.prototype.resolvedOptions = jest.fn().mockReturnValue({
                timeZone: 'America/New_York',
            });
        });

        afterAll(() => {
            jest.clearAllMocks();
        });

        it('should return columns for ActivityLogsContainerEnum.TRIGGER', () => {
            const columns = getTableColumns(ActivityLogsContainerEnum.TRIGGER);

            expect(columns).toEqual([
                {
                    id: 'name',
                    label: 'Performed by',
                    sortable: false,
                    truncate: true,
                    width: '20%',
                },
                {
                    id: 'message',
                    label: 'Activity',
                    sortable: false,
                },
                {
                    id: 'details',
                    label: 'Details',
                    sortable: false,
                    width: '150px',
                },
                {
                    id: 'eventTime',
                    label: 'Time (New York)',
                    sortable: true,
                    width: '150px',
                },
            ]);
        });

        it('should return columns for other containers', () => {
            const columns = getTableColumns(ActivityLogsContainerEnum.DELIVERY);

            expect(columns).toEqual([
                { id: 'name', label: 'Performed by', truncate: true, sortable: false, width: '20%' },
                { id: 'message', label: 'Activity', sortable: false },
                { id: 'targetType', label: 'Target', sortable: false, width: '200px' },
                { id: 'eventTime', label: 'Time (New York)', sortable: true, width: '150px' },
            ]);
        });
    });

    describe('getActivityLogColumnMessage', () => {
        it('should remove the label and return the modified message with lowercase first letter', () => {
            const label = 'Task';
            const message = 'Task completed successfully.';
            const result = getActivityLogColumnMessage(label, message);
            expect(result).toBe('completed successfully.');
        });

        it('should return the original message with lowercase first letter when label is not in message', () => {
            const label = 'Label';
            const message = 'This message does not contain the label.';
            const result = getActivityLogColumnMessage(label, message);
            expect(result).toBe('this message does not contain the label.');
        });

        it('should return empty string when message is only the label and there are no other characters', () => {
            const label = 'Task';
            const message = 'Task';
            const result = getActivityLogColumnMessage(label, message);
            expect(result).toBe('');
        });

        it('should handle case where message starts with the label and has leading/trailing spaces', () => {
            const label = 'Task';
            const message = '   Task was not completed.   ';
            const result = getActivityLogColumnMessage(label, message);
            expect(result).toBe('was not completed.');
        });

        it('should not modify message if label is an empty string', () => {
            const label = '';
            const message = 'Message without label';
            const result = getActivityLogColumnMessage(label, message);
            expect(result).toBe('message without label');
        });

        it('should handle empty message input gracefully', () => {
            const label = 'Task';
            const message = '';
            const result = getActivityLogColumnMessage(label, message);
            expect(result).toBe('');
        });
    });

    describe('calculateActivityLogColumnProps', () => {
        it('should return props for PHASE_COMPLETED', () => {
            const mockMessage = "Completed phase 'New phase'";
            const result = calculateActivityLogColumnProps('PHASE_COMPLETED', mockMessage);
            expect(result).toEqual({
                background: colors.successBackground,
                border: colors.green,
                label: 'Completed',
                message: "phase 'New phase'",
            });
        });

        it('should return props for RELEASE_COMPLETED', () => {
            const mockMessage = "Completed release 'New release'";
            const result = calculateActivityLogColumnProps('RELEASE_COMPLETED', mockMessage);
            expect(result).toEqual({
                background: colors.successBackground,
                border: colors.green,
                label: 'Completed',
                message: "release 'New release'",
            });
        });

        it('should return props for WORKFLOW_EXECUTION_COMPLETED', () => {
            const mockMessage = "Completed workflow 'New workflow'";
            const result = calculateActivityLogColumnProps('WORKFLOW_EXECUTION_COMPLETED', mockMessage);
            expect(result).toEqual({
                background: colors.successBackground,
                border: colors.green,
                label: 'Completed',
                message: "workflow 'New workflow'",
            });
        });

        it('should return props for DELIVERY_COMPLETED', () => {
            const mockMessage = "Completed delivery 'New delivery'";
            const result = calculateActivityLogColumnProps('DELIVERY_COMPLETED', mockMessage);
            expect(result).toEqual({
                background: colors.successBackground,
                border: colors.green,
                label: 'Completed',
                message: "delivery 'New delivery'",
            });
        });

        it('should return props for STAGE_COMPLETED', () => {
            const mockMessage = "Completed stage 'New stage'";
            const result = calculateActivityLogColumnProps('STAGE_COMPLETED', mockMessage);
            expect(result).toEqual({
                background: colors.successBackground,
                border: colors.green,
                label: 'Completed',
                message: "stage 'New stage'",
            });
        });

        it('should return props for ITEM_COMPLETED', () => {
            const mockMessage = "Completed tracked item 'New item' in stage 'stage 1' from release 'release1'";
            const result = calculateActivityLogColumnProps('ITEM_COMPLETED', mockMessage);
            expect(result).toEqual({
                background: colors.successBackground,
                border: colors.green,
                label: 'Completed',
                message: "tracked item 'New item' in stage 'stage 1' from release 'release1'",
            });
        });

        it('should return props for ITEM_COMPLETED_MANUALLY', () => {
            const mockMessage = "Completed tracked item 'New item' in stage 'stage 1' from release 'release1' manually";
            const result = calculateActivityLogColumnProps('ITEM_COMPLETED_MANUALLY', mockMessage);
            expect(result).toEqual({
                background: colors.successBackground,
                border: colors.green,
                label: 'Completed',
                message: "tracked item 'New item' in stage 'stage 1' from release 'release1' manually",
            });
        });

        it('should return props for TASK_COMPLETED', () => {
            const mockMessage = "Completed task 'New task'";
            const result = calculateActivityLogColumnProps('TASK_COMPLETED', mockMessage);
            expect(result).toEqual({
                background: colors.successBackground,
                border: colors.green,
                label: 'Completed',
                message: "task 'New task'",
            });
        });

        it('should return props for TASK_COMPLETED_IN_ADVANCE', () => {
            const mockMessage = "Completed in advance Task 'My task 2'";
            const result = calculateActivityLogColumnProps('TASK_COMPLETED_IN_ADVANCE', mockMessage);
            expect(result).toEqual({
                background: colors.successBackground,
                border: colors.green,
                label: 'Completed',
                message: "in advance Task 'My task 2'",
            });
        });

        it('should return props for TASK_SKIPPED', () => {
            const mockMessage = "Skipped Task 'My task 3'";
            const result = calculateActivityLogColumnProps('TASK_SKIPPED', mockMessage);
            expect(result).toEqual({
                background: colors.background,
                border: colors.lightgray,
                label: 'Skipped',
                message: "task 'My task 3'",
            });
        });

        it('should return props for TASK_SKIPPED_IN_ADVANCE', () => {
            const mockMessage = "Skipped in advance Task 'My task 3'";
            const result = calculateActivityLogColumnProps('TASK_SKIPPED_IN_ADVANCE', mockMessage);
            expect(result).toEqual({
                background: colors.background,
                border: colors.lightgray,
                label: 'Skipped',
                message: "in advance Task 'My task 3'",
            });
        });

        it('should return props for TASK_FAILED', () => {
            const mockMessage = "Failed task 'New task'";
            const result = calculateActivityLogColumnProps('TASK_FAILED', mockMessage);
            expect(result).toEqual({
                background: colors.errorBackground,
                border: colors.red,
                label: 'Failed',
                message: "task 'New task'",
            });
        });

        it('should return props for PHASE_FAILED', () => {
            const mockMessage = "Failed Phase 'New phase'";
            const result = calculateActivityLogColumnProps('PHASE_FAILED', mockMessage);
            expect(result).toEqual({
                background: colors.errorBackground,
                border: colors.red,
                label: 'Failed',
                message: "phase 'New phase'",
            });
        });

        it('should return props for RELEASE_FAILED', () => {
            const mockMessage = 'Failed Release';
            const result = calculateActivityLogColumnProps('RELEASE_FAILED', mockMessage);
            expect(result).toEqual({
                background: colors.errorBackground,
                border: colors.red,
                label: 'Failed',
                message: 'release',
            });
        });

        it('should return props for WORKFLOW_EXECUTION_FAILED', () => {
            const mockMessage = 'Failed Workflow execution';
            const result = calculateActivityLogColumnProps('WORKFLOW_EXECUTION_FAILED', mockMessage);
            expect(result).toEqual({
                background: colors.errorBackground,
                border: colors.red,
                label: 'Failed',
                message: 'workflow execution',
            });
        });

        it('should return props for TRIGGER_FAILED', () => {
            const mockMessage = "Failed trigger 'trigger 1' due to error: Unknown error";
            const result = calculateActivityLogColumnProps('TRIGGER_FAILED', mockMessage);
            expect(result).toEqual({
                background: colors.errorBackground,
                border: colors.red,
                label: 'Failed',
                message: "trigger 'trigger 1' due to error: Unknown error",
            });
        });

        it('should return props for TRIGGER_FILTER_FAILED', () => {
            const mockMessage = "Failed trigger 'Trigger 1' filter due to error: Unknown error";
            const result = calculateActivityLogColumnProps('TRIGGER_FILTER_FAILED', mockMessage);
            expect(result).toEqual({
                background: colors.errorBackground,
                border: colors.red,
                label: 'Failed',
                message: "trigger 'Trigger 1' filter due to error: Unknown error",
            });
        });

        it('should return null for an unknown activityType', () => {
            const result = calculateActivityLogColumnProps('UNKNOWN_TYPE', 'Some message');
            expect(result).toBeNull();
        });
    });

    describe('getTargetTypeLabel', () => {
        it('should return "Release" for "xlrelease.Release" when containerEnum is RELEASE', () => {
            const result = getTargetTypeLabel('xlrelease.Release', ActivityLogsContainerEnum.RELEASE);
            expect(result).toBe('Release');
        });

        it('should return "Phase" for "xlrelease.Phase" when containerEnum is RELEASE', () => {
            const result = getTargetTypeLabel('xlrelease.Phase', ActivityLogsContainerEnum.RELEASE);
            expect(result).toBe('Phase');
        });

        it('should return "Team" for "xlrelease.Team" when containerEnum is RELEASE', () => {
            const result = getTargetTypeLabel('xlrelease.Team', ActivityLogsContainerEnum.RELEASE);
            expect(result).toBe('Team');
        });

        it('should return "Variable" for all variable types when containerEnum is RELEASE', () => {
            const variableTypes = [
                'xlrelease.StringVariable',
                'xlrelease.PasswordStringVariable',
                'xlrelease.BooleanVariable',
                'xlrelease.IntegerVariable',
                'xlrelease.ListStringVariable',
                'xlrelease.ReferenceVariable',
                'xlrelease.DateVariable',
                'xlrelease.MapStringStringVariable',
                'xlrelease.SetStringVariable',
            ];
            variableTypes.forEach((type) => {
                expect(getTargetTypeLabel(type, ActivityLogsContainerEnum.RELEASE)).toBe('Variable');
            });
        });

        it('should return "Attachment" for "xlrelease.Attachment" when containerEnum is RELEASE', () => {
            const result = getTargetTypeLabel('xlrelease.Attachment', ActivityLogsContainerEnum.RELEASE);
            expect(result).toBe('Attachment');
        });

        it('should return "Task" for "xlrelease.Comment" when containerEnum is RELEASE', () => {
            const result = getTargetTypeLabel('xlrelease.Comment', ActivityLogsContainerEnum.RELEASE);
            expect(result).toBe('Task');
        });

        it('should return "Template logo" for "xlrelease.TemplateLogo" when containerEnum is RELEASE', () => {
            const result = getTargetTypeLabel('xlrelease.TemplateLogo', ActivityLogsContainerEnum.RELEASE);
            expect(result).toBe('Template logo');
        });

        it('should return "Folder" for "xlrelease.Folder" when containerEnum is RELEASE', () => {
            const result = getTargetTypeLabel('xlrelease.Folder', ActivityLogsContainerEnum.RELEASE);
            expect(result).toBe('Folder');
        });

        it('should return "Task" for an unknown targetType when containerEnum is RELEASE', () => {
            const result = getTargetTypeLabel('unknown.Type', ActivityLogsContainerEnum.RELEASE);
            expect(result).toBe('Task');
        });
        it('should return "Delivery" for "delivery.Delivery" when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('delivery.Delivery', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('Delivery');
        });

        it('should return "Stage" for "delivery.Stage" when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('delivery.Stage', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('Stage');
        });

        it('should return "TrackedItem" for "delivery.TrackedItem" when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('delivery.TrackedItem', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('Tracked item');
        });

        it('should return "StageTrackedItem" for "delivery.StageTrackedItem" when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('delivery.StageTrackedItem', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('Stage tracked item');
        });

        it('should return "Transition" for "delivery.Transition" when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('delivery.Transition', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('Transition');
        });

        it('should return "Condition" for "delivery.Condition" when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('delivery.Condition', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('Condition');
        });

        it('should return "Condition" for "delivery.ItemsCompletionCondition" when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('delivery.ItemsCompletionCondition', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('Condition');
        });

        it('should return "Condition" for "delivery.ConditionGroup" when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('delivery.ConditionGroup', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('Condition');
        });

        it('should return "Condition" for "delivery.TimeCondition" when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('delivery.TimeCondition', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('Condition');
        });

        it('should return the targetType itself for an unknown targetType when containerEnum is not RELEASE', () => {
            const result = getTargetTypeLabel('unknown.Type', ActivityLogsContainerEnum.DELIVERY);
            expect(result).toBe('unknown.Type');
        });
    });

    describe('getTaskWithOpenedModalHref', () => {
        afterAll(() => {
            // Reset the mock before each test to clear previous call data
            jest.clearAllMocks();
        });

        it('should return the correct URL with the release view segment, releaseId, and phaseTaskId', () => {
            const releaseId = 'release-123';
            const phaseTaskId = 'task-456';
            window.location.href = '#/releases/Folder1-Release1';

            const result = getTaskWithOpenedModalHref(releaseId, phaseTaskId);
            expect(result).toBe(`#/releases/${releaseId}?openTaskDetailsModal=${phaseTaskId}`);
        });
    });

    describe('getReleaseHref', () => {
        it('should return the correct URL with the release view segment, releaseId, and phaseTaskId', () => {
            const releaseId = 'release-123';
            window.location.href = '#/releases/Folder1-Release1';

            const result = getReleaseHref(releaseId);
            expect(result).toBe(`#/releases/${releaseId}`);
        });
    });

    describe('getReleaseHref', () => {
        it('should return the correct URL with the release view segment and releaseId', () => {
            const releaseId = 'release-123';
            window.location.href = '#/releases/Folder1-Release1';

            const result = getReleaseHref(releaseId);
            expect(result).toBe(`#/releases/${releaseId}`);
        });
    });

    describe('getDeliveryUrlFromPath', () => {
        it('should return the correct URL for delivery patterns', () => {
            window.location.href =
                '#/folders/Applications/Folderd70a3bbce22548c791e6f7bc0aa645b6/delivery-patterns/Deliveries/Delivery704722cdd5644d1dbfed589fa540cb2e/logs?has_filter&isImportantHighlighted';
            expect(getDeliveryUrlFromPath('Deliveries/Delivery704722cdd5644d1dbfed589fa540cb2e')).toBe(
                '#/folders/Applications/Folderd70a3bbce22548c791e6f7bc0aa645b6/delivery-patterns/Deliveries/Delivery704722cdd5644d1dbfed589fa540cb2e',
            );
        });
        it('should return the correct URL for deliveries', () => {
            window.location.href = '#/deliveries/Deliveries/Delivery560a196c6d5a47f29b20b9d3250f4a4b/logs?has_filter&isImportantHighlighted';
            expect(getDeliveryUrlFromPath('Deliveries/Delivery560a196c6d5a47f29b20b9d3250f4a4b')).toBe(
                '#/deliveries/Deliveries/Delivery560a196c6d5a47f29b20b9d3250f4a4b',
            );
        });
    });

    describe('generateIdFromUsername', () => {
        it('should generate an ID with lowercase and dash-separated words from a simple username', () => {
            const result = generateIdFromUsername('Task Comment');
            expect(result).toBe('id-task-comment');
        });

        it('should remove special characters and generate a clean ID', () => {
            const result = generateIdFromUsername('Task!@#Comment$$$');
            expect(result).toBe('id-taskcomment');
        });

        it('should replace multiple spaces with a single dash', () => {
            const result = generateIdFromUsername('Task    Comment');
            expect(result).toBe('id-task-comment');
        });

        it('should trim leading and trailing spaces', () => {
            const result = generateIdFromUsername('   Task Comment   ');
            expect(result).toBe('id-task-comment');
        });

        it('should convert uppercase letters to lowercase', () => {
            const result = generateIdFromUsername('TASK COMMENT');
            expect(result).toBe('id-task-comment');
        });

        it('should handle empty input gracefully', () => {
            const result = generateIdFromUsername('');
            expect(result).toBe('id-');
        });

        it('should handle a username with only special characters gracefully', () => {
            const result = generateIdFromUsername('!@#$$%^&*()');
            expect(result).toBe('id-');
        });

        it('should handle a username with only spaces gracefully', () => {
            const result = generateIdFromUsername('     ');
            expect(result).toBe('id-');
        });
    });

    describe('removeAllAfterLastTask', () => {
        it('should return the matched part of the task ID before the last occurrence of "Task"', () => {
            const taskId = 'Phase1/Task123/Comment1';
            const result = removeAllAfterLastTask(taskId);
            expect(result).toBe('Phase1/Task123');
        });

        it('should return the task ID unchanged if it does not match the pattern', () => {
            const taskId = 'Phase1/Attachment1';
            const result = removeAllAfterLastTask(taskId);
            expect(result).toBe(taskId);
        });

        it('should handle case with multiple tasks', () => {
            const taskId = 'Phase1/Task1/Task2/Task3/Comment1';
            const result = removeAllAfterLastTask(taskId);
            expect(result).toBe('Phase1/Task1/Task2/Task3');
        });

        it('should handle an empty string as input and return an empty string', () => {
            const taskId = '';
            const result = removeAllAfterLastTask(taskId);
            expect(result).toBe('');
        });
    });

    describe('getSystemUserAvatar', () => {
        it('should return RELEASE_LOGO_SVG when username is SYSTEM_USER', () => {
            const result = getSystemUserAvatar(SYSTEM_USER);
            expect(result).toBe(RELEASE_LOGO_SVG);
        });

        it('should return undefined when username is not SYSTEM_USER', () => {
            const result = getSystemUserAvatar('someOtherUser');
            expect(result).toBeUndefined();
        });
    });

    describe('checkIfImportantActivity', () => {
        it('should return true for important activity types', () => {
            IMPORTANT_ACTIVITY_TYPES.forEach((activityType) => {
                expect(checkIfImportantActivity(activityType)).toBe(true);
            });
        });

        it('should return false for non-important activity types', () => {
            const nonImportantTypes = ['NON_EXISTENT_ACTIVITY', 'RANDOM_EVENT', 'TASK_UNKNOWN'];

            nonImportantTypes.forEach((activityType) => {
                expect(checkIfImportantActivity(activityType)).toBe(false);
            });
        });

        it('should return false for an empty string', () => {
            expect(checkIfImportantActivity('')).toBe(false);
        });

        it('should return false for undefined or null', () => {
            expect(checkIfImportantActivity(undefined as never)).toBe(false);
            expect(checkIfImportantActivity(null as never)).toBe(false);
        });
    });

    describe('getFilterSettingsForLoadPayload', () => {
        it('should return filters without fields for payload', () => {
            expect(getFilterSettingsForLoadPayload({ fakeKey: 'fakeValue', folderId: 'Folder1', isImportantHighlighted: true } as never)).toStrictEqual({
                fakeKey: 'fakeValue',
            });
        });
    });

    describe('checkIfMessageIsMarkdown', () => {
        it('should return true for TRIGGER_EXECUTED with a release link in the message', () => {
            const result = checkIfMessageIsMarkdown('TRIGGER_EXECUTED', 'Check this link: #/releases/123');
            expect(result).toBe(true);
        });

        it('should return true for TRIGGER_SKIPPED with a template link in the message', () => {
            const result = checkIfMessageIsMarkdown('TRIGGER_SKIPPED', 'Check this link: #/templates/123');
            expect(result).toBe(true);
        });

        it('should return false for TRIGGER_EXECUTED without a release or template link in the message', () => {
            const result = checkIfMessageIsMarkdown('TRIGGER_EXECUTED', 'No relevant link here');
            expect(result).toBe(false);
        });

        it('should return false for an activity type other than TRIGGER_EXECUTED or TRIGGER_SKIPPED', () => {
            const result = checkIfMessageIsMarkdown('OTHER_TYPE', 'Check this link: #/releases/123');
            expect(result).toBe(false);
        });

        it('should return false for TRIGGER_SKIPPED without a release or template link in the message', () => {
            const result = checkIfMessageIsMarkdown('TRIGGER_SKIPPED', 'No relevant link here');
            expect(result).toBe(false);
        });
    });
});
