import { all, call, put, select, takeLatest } from 'redux-saga/effects';
import { SagaIterator } from 'redux-saga';
import { activityLogs, getActivityLogsState } from './activity-logs.reducer';
import activityLogsSaga, {
    getFiltersWithLogUser,
    cleanLogFilters,
    initSaga,
    loadFilteredAndPagedLogsSaga,
    loadLogs,
    replaceLogUserWithSystemUser,
    updateClientSettingsAndUrl,
    setHighlightImportantSaga,
    loadEventDialogSaga,
} from './activity-logs.saga';
import { ActivityLog, ActivityLogsContainerEnum } from '../types';
import { httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { DEFAULT_FILTER_SETTINGS, LOG_USER, LOGS_ITEMS_PER_PAGE, SYSTEM_USER } from '../constants';
import { User } from '../../../../../../../../../core/xlr-ui/app/types';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { getFilterSettingsForLoadPayload } from '../helper';

const {
    init,
    loadEventDialog,
    loadFilteredAndPagedLogs,
    setActivityTypes,
    setEventDetails,
    setEventDialogOpened,
    setHighlightImportant,
    setIsLastPage,
    setIsLoading,
    setLogs,
    setLogsFilter,
    setPage,
    setUsers,
} = activityLogs.actions;

describe('activity logs saga', () => {
    const activityTypes: string[] = ['ATTACHMENT_ADDED_ON_TASK', 'COMMENT_ADDED'];
    const users: User[] = [{ username: 'admin' }, { username: SYSTEM_USER }];
    const logs: ActivityLog[] = [
        {
            user: {
                username: 'admin',
                fullName: 'Release Administrator',
                email: 'releaseadmin@dai.com',
            },
            activityType: 'RELEASE_STARTED',
            message: 'Release started',
            eventTime: 1726472112657,
            targetType: 'time.Schedule',
            targetId: 'Applications/FolderSamplesAndTutorials/Triggerf5d982a3d9724f8888e28167cd68ebef',
            dataId: null,
        },
    ];

    const logsFilterMock = {
        containerEnum: ActivityLogsContainerEnum.RELEASE,
        containerId: '123',
        filterSettings: { ...DEFAULT_FILTER_SETTINGS, usernames: [SYSTEM_USER, 'admin', 'i do not exist'] },
    };

    it('should yield all effects', () => {
        const gen: SagaIterator = activityLogsSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(init, initSaga),
                takeLatest(loadFilteredAndPagedLogs, loadFilteredAndPagedLogsSaga),
                takeLatest(setHighlightImportant, setHighlightImportantSaga),
                takeLatest(loadEventDialog, loadEventDialogSaga),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('initSaga', () => {
        const cleanedLogFilters = { ...logsFilterMock, filterSettings: { ...DEFAULT_FILTER_SETTINGS, usernames: [SYSTEM_USER, 'admin'] } };

        it('should call proper generators on init saga', () => {
            const generator: SagaIterator = initSaga({ payload: logsFilterMock, type: 'INIT' });
            expect(generator.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(generator.next().value).toStrictEqual(put(setPage(0)));
            expect(generator.next().value).toStrictEqual(all([call(httpGET, 'activity/123/logs/activity-types'), call(httpGET, 'activity/123/logs/users')]));
            expect(generator.next([{ data: activityTypes }, { data: users }]).value).toStrictEqual(put(setActivityTypes(activityTypes)));
            expect(generator.next().value).toStrictEqual(put(setUsers(users)));
            expect(generator.next().value).toStrictEqual(put(setLogsFilter(cleanedLogFilters)));
            expect(generator.next().value).toStrictEqual(call(loadLogs, 0, cleanedLogFilters));
            expect(generator.next(logs).value).toStrictEqual(put(setIsLastPage(true)));
            expect(generator.next().value).toStrictEqual(put(setLogs(logs)));
            expect(generator.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(generator.next().done).toBe(true);
        });
    });

    describe('loadFilteredAndPagedLogsSaga', () => {
        const oldLogs: ActivityLog[] = [
            {
                user: {
                    username: 'admin',
                    fullName: 'Test',
                    email: 'releaseadmin@dai.com',
                },
                activityType: 'RELEASE_STARTED',
                message: 'Release started',
                eventTime: 1726472112655,
                targetType: 'time.Schedule',
                targetId: 'Applications/FolderSamplesAndTutorials/Triggerf5d982a3d9724f8888e28167cd68ebef',
                dataId: null,
            },
        ];

        it('should call proper generators when filters provided', () => {
            const logsFilter = { test: 'test' } as never;
            const generator: SagaIterator = loadFilteredAndPagedLogsSaga({ payload: { logsFilter, page: 0 }, type: 'LOAD_FILTERED_AND_PAGED_LOGS_SAGA' });
            expect(generator.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(generator.next().value).toStrictEqual(put(setPage(0)));
            expect(generator.next().value).toStrictEqual(select(getActivityLogsState));
            expect(generator.next({ logs: oldLogs }).value).toStrictEqual(put(setLogsFilter(logsFilter)));
            expect(generator.next().value).toStrictEqual(call(loadLogs, 0, logsFilter));
            expect(generator.next(logs).value).toStrictEqual(put(setIsLastPage(true)));
            expect(generator.next().value).toStrictEqual(put(setLogs(logs)));
            expect(generator.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(generator.next().done).toBe(true);
        });

        it('should proper generators when not on the first page', () => {
            const logsFilter = { test: 'test' } as never;
            const page = 1;
            const generator: SagaIterator = loadFilteredAndPagedLogsSaga({ payload: { logsFilter, page }, type: 'LOAD_FILTERED_AND_PAGED_LOGS_SAGA' });
            expect(generator.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(generator.next().value).toStrictEqual(put(setPage(page)));
            expect(generator.next().value).toStrictEqual(select(getActivityLogsState));
            expect(generator.next({ logs: oldLogs }).value).toStrictEqual(put(setLogsFilter(logsFilter)));
            expect(generator.next().value).toStrictEqual(call(loadLogs, page, logsFilter));
            expect(generator.next(logs).value).toStrictEqual(put(setIsLastPage(true)));
            expect(generator.next().value).toStrictEqual(put(setLogs(oldLogs.concat(logs))));
            expect(generator.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(generator.next().done).toBe(true);
        });
    });

    describe('loadEventDialogSaga', () => {
        it('should call proper generators', () => {
            const generator: SagaIterator = loadEventDialogSaga({ payload: 'test', type: 'LOAD_EVENT_DIALOG' });
            expect(generator.next().value).toStrictEqual(put(setEventDialogOpened(true)));
            expect(generator.next().value).toStrictEqual(call(httpGET, 'activity/data/test'));
            expect(generator.next({ data: { response: 'ok', content: '{"id": 1}' } }).value).toStrictEqual(
                put(setEventDetails('{\n  "response": "ok",\n  "content": {\n    "id": 1\n  }\n}')),
            );
            expect(generator.next().done).toBe(true);
        });
        it('should ignore JSON parse errors', () => {
            const generator: SagaIterator = loadEventDialogSaga({ payload: 'test', type: 'LOAD_EVENT_DIALOG' });
            expect(generator.next().value).toStrictEqual(put(setEventDialogOpened(true)));
            expect(generator.next().value).toStrictEqual(call(httpGET, 'activity/data/test'));
            expect(generator.next({ data: { response: 'ok', content: '{"aint json here}"' } }).value).toStrictEqual(
                put(setEventDetails('{\n  "response": "ok",\n  "content": "{\\"aint json here}\\""\n}')),
            );
            expect(generator.next().done).toBe(true);
        });
    });

    describe('loadLogs Saga', () => {
        const filterSettingsWithTaskComment = getFiltersWithLogUser(logsFilterMock);
        it('should call proper generators', () => {
            const page = 1;
            const generator: SagaIterator = loadLogs(1, logsFilterMock);
            expect(generator.next().value).toEqual(
                call(
                    httpPOST,
                    `activity/${logsFilterMock.containerId}/logs/search?page=${page}&resultsPerPage=${LOGS_ITEMS_PER_PAGE}`,
                    getFilterSettingsForLoadPayload(filterSettingsWithTaskComment.filterSettings),
                ),
            );
            expect(generator.next(logs).value).toStrictEqual(call(updateClientSettingsAndUrl, logsFilterMock.containerId, logsFilterMock.filterSettings));
            expect(generator.next().done).toBe(true);
        });
    });

    describe('setHighlightImportantSaga', () => {
        it('should call proper generators', () => {
            const logsFilter = logsFilterMock;
            const newFilterSettings = { ...logsFilter.filterSettings, isImportantHighlighted: true };
            const generator: SagaIterator = setHighlightImportantSaga({ payload: true, type: 'SET_HIGHLIGHT_IMPORTANT_SAGA' });
            expect(generator.next().value).toStrictEqual(select(getActivityLogsState));
            expect(generator.next({ logsFilter }).value).toStrictEqual(put(setLogsFilter({ ...logsFilter, filterSettings: newFilterSettings })));
            expect(generator.next(logs).value).toStrictEqual(call(updateClientSettingsAndUrl, logsFilterMock.containerId, newFilterSettings));
            expect(generator.next().done).toBe(true);
        });
    });

    describe('cleanLogFilters', () => {
        it('should remove invalid filters', () => {
            const logsFilter = {
                containerEnum: ActivityLogsContainerEnum.RELEASE,
                containerId: '123',
                filterSettings: {
                    usernames: ['admin', 'invalidUser'],
                    activityTypes: ['ATTACHMENT_ADDED_ON_TASK', 'INVALID_TYPE'],
                },
            } as never;

            const cleanedFilters = cleanLogFilters(logsFilter, activityTypes, users);

            expect(cleanedFilters.filterSettings.usernames).toEqual(['admin']);
            expect(cleanedFilters.filterSettings.activityTypes).toEqual(['ATTACHMENT_ADDED_ON_TASK']);
        });
    });

    describe('replaceLogUserWithSystemUser', () => {
        it('should replace LOG_USER with SYSTEM_USER if SYSTEM_USER is not present', () => {
            const result = replaceLogUserWithSystemUser([{ username: LOG_USER }]);
            expect(result).toEqual([{ username: SYSTEM_USER }]);
        });

        it('should remove LOG_USER if SYSTEM_USER is present', () => {
            const result = replaceLogUserWithSystemUser([{ username: LOG_USER }, { username: SYSTEM_USER }]);
            expect(result).toEqual([{ username: SYSTEM_USER }]);
        });

        it('should return the same users if LOG_USER is not present', () => {
            const result = replaceLogUserWithSystemUser([{ username: 'admin' }]);
            expect(result).toEqual([{ username: 'admin' }]);
        });
    });

    describe('getFiltersWithLogUser', () => {
        it('should add LOG_USER if SYSTEM_USER is present', () => {
            const logsFilter = {
                filterSettings: {
                    usernames: [SYSTEM_USER],
                },
            } as never;

            const result = getFiltersWithLogUser(logsFilter);
            expect(result.filterSettings.usernames).toEqual([SYSTEM_USER, LOG_USER]);
        });

        it('should not modify usernames if SYSTEM_USER is not present', () => {
            const logsFilter = {
                filterSettings: {
                    usernames: ['admin'],
                },
            } as never;

            const result = getFiltersWithLogUser(logsFilter);
            expect(result.filterSettings.usernames).toEqual(['admin']);
        });
    });

    describe('updateClientSettingsAndUrl', () => {
        const FiltersQueryParams = {
            update: jest.fn(),
        };
        const ClientSettings = {
            setLogsFilters: jest.fn(),
        };
        it('should call proper generators ', () => {
            const generator: SagaIterator = updateClientSettingsAndUrl(logsFilterMock.containerId, logsFilterMock.filterSettings);
            expect(generator.next().value).toStrictEqual(call(getAngularService, 'FiltersQueryParams'));
            expect(generator.next(FiltersQueryParams).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(generator.next(ClientSettings).done).toBe(true);
            expect(FiltersQueryParams.update).toHaveBeenCalledWith(logsFilterMock.filterSettings);
            expect(ClientSettings.setLogsFilters).toHaveBeenCalledWith(logsFilterMock.containerId, logsFilterMock.filterSettings);
        });
    });
});
