package com.xebialabs.xlrelease.api.internal;

import java.util.List;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.ReleaseActivityLogsService;
import com.xebialabs.xlrelease.views.ActivityLogEntryView;
import com.xebialabs.xlrelease.views.LogsFilters;
import com.xebialabs.xlrelease.views.converters.UserViewConverter;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.api.ApiService.DEFAULT_RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.api.ApiService.PAGE;
import static com.xebialabs.xlrelease.api.ApiService.RESULTS_PER_PAGE;

@Path("/activity")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class ReleaseActivityLogsResource extends BaseActivityLogsResource {

    private PermissionChecker permissions;

    @Autowired
    public ReleaseActivityLogsResource(
            final ReleaseActivityLogsService releaseActivityLogsService,
            final UserViewConverter userViewConverter,
            final PermissionChecker permissions) {
        super(releaseActivityLogsService, userViewConverter);
        this.permissions = permissions;
    }

    /**
     * Searches in the activity logs of a release.
     *
     * @param releaseId the identifier of the release
     * @param filters   the search criteria to use
     * @return the activity logs
     */
    @POST
    @Path("/{releaseId:.*Release[^/-]*}/logs/search")
    public List<ActivityLogEntryView> searchReleaseLogs(
            @PathParam("releaseId") @IdParam String releaseId,
            @DefaultValue("0") @QueryParam(PAGE) int page,
            @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) int resultsPerPage,
            LogsFilters filters
    ) {
        checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than %d", DEFAULT_RESULTS_PER_PAGE);
        permissions.checkView(releaseId);

        // sort feature of Pageable is not used, because there is sort order in LogsFilters
        PageRequest pageable = PageRequest.of(page, resultsPerPage, Sort.unsorted());
        return searchLogs(releaseId, filters, pageable);
    }

}
