package com.xebialabs.xlrelease.repository

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.activity.ActivityOps
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.db.sql.{LimitOffset, SqlBuilder}
import com.xebialabs.xlrelease.domain.ActivityCategory._
import com.xebialabs.xlrelease.domain.{ActivityCategory, ActivityLogEntry}
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.ActivityLogSchema.ACTIVITY_LOGS
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{ActivityLogSchema, ActivityLogSqlBuilder, PersistenceSupport}
import com.xebialabs.xlrelease.views.LogsFilters
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.data.domain.Pageable
import org.springframework.jdbc.core.namedparam.{MapSqlParameterSource, SqlParameterSource}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import java.util
import java.util.Date
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

@Repository
@IsTransactional
class SqlActivityLogRepository @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                            @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect)
  extends ActivityLogRepository with PersistenceSupport with LimitOffset {

  @Timed
  override def log(containerId: String, activityLogEntries: util.List[ActivityLogEntry], username: String): Unit = {
    namedTemplate.batchUpdate(
      s"""INSERT INTO ${ACTIVITY_LOGS.TABLE} (
         |  ${ACTIVITY_LOGS.CONTAINER_ID},
         |  ${ACTIVITY_LOGS.USERNAME},
         |  ${ACTIVITY_LOGS.ACTIVITY_TYPE},
         |  ${ACTIVITY_LOGS.MESSAGE},
         |  ${ACTIVITY_LOGS.EVENT_TIME},
         |  ${ACTIVITY_LOGS.TARGET_TYPE},
         |  ${ACTIVITY_LOGS.TARGET_ID},
         |  ${ACTIVITY_LOGS.DATA_ID}
         | ) VALUES (
         |  :${ACTIVITY_LOGS.CONTAINER_ID},
         |  :${ACTIVITY_LOGS.USERNAME},
         |  :${ACTIVITY_LOGS.ACTIVITY_TYPE},
         |  :${ACTIVITY_LOGS.MESSAGE},
         |  :${ACTIVITY_LOGS.EVENT_TIME},
         |  :${ACTIVITY_LOGS.TARGET_TYPE},
         |  :${ACTIVITY_LOGS.TARGET_ID},
         |  :${ACTIVITY_LOGS.DATA_ID}
         | )""".stripMargin,
      activityLogEntries.asScala.map(entry => {
        val params = new MapSqlParameterSource()
        params.addValue(ACTIVITY_LOGS.CONTAINER_ID, containerId.toContainerName)
        params.addValue(ACTIVITY_LOGS.USERNAME, username)
        params.addValue(ACTIVITY_LOGS.ACTIVITY_TYPE, entry.getActivityType)
        params.addValue(ACTIVITY_LOGS.MESSAGE, entry.getMessage.truncateBytes(ActivityLogSchema.MAX_COLUMN_LENGTH))
        params.addValue(ACTIVITY_LOGS.EVENT_TIME, entry.getEventTime.asTimestamp)
        params.addValue(ACTIVITY_LOGS.TARGET_TYPE, entry.getTargetType)
        params.addValue(ACTIVITY_LOGS.TARGET_ID, entry.getTargetId)
        params.addValue(ACTIVITY_LOGS.DATA_ID, entry.getDataId)
        params
      }).toArray[SqlParameterSource]
    )
  }

  @Timed
  override def findAllLogsOf(containerId: ContainerId,
                             logsFilters: LogsFilters,
                             knownActivityOps: java.util.List[_ <: ActivityOps],
                             pageable: Pageable): util.List[ActivityLogEntry] = {
    val order = if (logsFilters == null) {
      "ASC"
    } else {
      SqlBuilder.getOrderAscOrDesc(Some(logsFilters.isDateAsc), default = true)
    }
    val sqlBuilder = new ActivityLogSqlBuilder()
      .select()
      .withContainer(containerId.toContainerName)
      .orderBy(s"${ACTIVITY_LOGS.EVENT_TIME} ${order}, ${ACTIVITY_LOGS.ID} ${order}")
    if (logsFilters != null) {
      val (activityTypes, otherExceptActivityTypes) = mapActivityTypes(logsFilters, knownActivityOps.asScala.toList)
      sqlBuilder
        .withFilter(logsFilters.getFilter)
        .withFrom(logsFilters.getFrom)
        .withTo(logsFilters.getTo)
        .withActivityTypes(activityTypes, otherExceptActivityTypes)
        .withTargetId(logsFilters.getTargetId)
    }
    query(sqlBuilder, pageable)
  }

  private def mapActivityTypes(filters: LogsFilters, knownActivityOps: List[_ <: ActivityOps]): (Seq[ActivityOps], Option[Seq[ActivityOps]]) = {
    val allowedCategories = ListBuffer.empty[ActivityCategory]

    if (filters.withImportant) allowedCategories += IMPORTANT
    if (filters.withReleaseEdit) allowedCategories += RELEASE_EDIT
    if (filters.withTaskEdit) allowedCategories += TASK_EDIT
    if (filters.withComments) allowedCategories += COMMENTS
    if (filters.withLifecycle) allowedCategories += LIFECYCLE
    if (filters.withReassign) allowedCategories += REASSIGN
    if (filters.withReportingRecordEdit) allowedCategories += REPORTING_RECORD_EDIT
    if (filters.withSecurity) allowedCategories += SECURITY
    if (filters.withDeliveryEdit) allowedCategories += DELIVERY_EDIT
    if (filters.withTriggerEdit) allowedCategories += TRIGGER_EDIT
    if (filters.withExecution) allowedCategories += EXECUTION
    if (filters.withOther) allowedCategories += OTHER

    val allowedCategoriesSet = allowedCategories.toSet
    val allowedOps = ListBuffer.empty[ActivityOps]
    knownActivityOps.foreach(op => {
      if (op.getCategories.asScala.exists(category => allowedCategoriesSet.contains(category))) {
        allowedOps += op
      }
    })
    val allowedOtherExcept: Option[Seq[ActivityOps]] = if (filters.withOther()) {
      Some(knownActivityOps)
    } else {
      None
    }
    (allowedOps.toSeq, allowedOtherExcept)
  }

  @Timed
  override def findAllLogsOf(containerId: ContainerId, activityOps: ActivityOps, pageable: Pageable): util.List[ActivityLogEntry] = {
    query(new ActivityLogSqlBuilder()
      .select()
      .withContainer(containerId.toContainerName)
      .withActivityType(activityOps)
      .orderBy(s"${ACTIVITY_LOGS.EVENT_TIME} ASC, ${ACTIVITY_LOGS.ID} ASC"),
      pageable)
  }

  private def query(activityLogSqlBuilder: ActivityLogSqlBuilder, pageable: Pageable): util.List[ActivityLogEntry] = {
    pageable.getSort.get().forEach(order => {
      throw new IllegalArgumentException("Ordering by Pageable is not supported, please use LogsFilters")
    })
    val (sql, params) = if (pageable.isPaged) {
      activityLogSqlBuilder.limitAndOffset(pageable.getPageSize, pageable.getOffset).build()
    } else {
      activityLogSqlBuilder.build()
    }
    jdbcTemplate.query(
      sql,
      params.toArray,
      new RowMapper[ActivityLogEntry] {
        override def mapRow(rs: ResultSet, rowNum: Int): ActivityLogEntry = {
          val activity = new ActivityLogEntry(rs.getString(ACTIVITY_LOGS.ACTIVITY_TYPE), rs.getString(ACTIVITY_LOGS.MESSAGE))
          activity.setId(rs.getString(ACTIVITY_LOGS.ID))
          activity.setEventTime(rs.getTimestamp(ACTIVITY_LOGS.EVENT_TIME))
          activity.setUsername(rs.getString(ACTIVITY_LOGS.USERNAME))
          activity.setTargetType(rs.getString(ACTIVITY_LOGS.TARGET_TYPE))
          activity.setTargetId(rs.getString(ACTIVITY_LOGS.TARGET_ID))
          activity.setDataId(rs.getString(ACTIVITY_LOGS.DATA_ID))
          activity
        }
      })
  }

  @Timed
  override def move(fromContainerId: String, toContainerId: String): Unit = {
    jdbcTemplate.update(s"UPDATE ${ACTIVITY_LOGS.TABLE} SET ${ACTIVITY_LOGS.CONTAINER_ID} = ? WHERE ${ACTIVITY_LOGS.CONTAINER_ID} = ?",
      toContainerId.toContainerName, fromContainerId.toContainerName)
  }

  private val STMT_DELETE_BY_CONTAINER_ID =
    s"""DELETE FROM ${ACTIVITY_LOGS.TABLE} WHERE ${ACTIVITY_LOGS.CONTAINER_ID} = :${ACTIVITY_LOGS.CONTAINER_ID}"""

  private val STMT_DELETE_BY_ACTIVITY_TYPE_AND_DATE =
    s"""DELETE FROM ${ACTIVITY_LOGS.TABLE} WHERE ${ACTIVITY_LOGS.ACTIVITY_TYPE} IN (:${ACTIVITY_LOGS.ACTIVITY_TYPE})
       | AND ${ACTIVITY_LOGS.EVENT_TIME} < :${ACTIVITY_LOGS.EVENT_TIME}""".stripMargin

  @Timed
  override def deleteByActivityTypesAndDate(activityTypes: java.util.List[ActivityOps], date: Date): Unit = {
    val activityTypesString = activityTypes.asScala.map(_.toString).asJava
    sqlUpdate(STMT_DELETE_BY_ACTIVITY_TYPE_AND_DATE, params(ACTIVITY_LOGS.ACTIVITY_TYPE -> activityTypesString,
      ACTIVITY_LOGS.EVENT_TIME -> date.asTimestamp), _ => ())
  }

  @Timed
  override def delete(containerId: CiId): Unit = {
    sqlUpdate(STMT_DELETE_BY_CONTAINER_ID, params(ACTIVITY_LOGS.CONTAINER_ID -> containerId.toContainerName), _ => ())
  }


  implicit class IdOps(id: ContainerId) {
    def toContainerName: String = getName(id.normalized)
  }

  type ContainerId = String

}
