package com.xebialabs.xlrelease.activity;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.xlrelease.domain.ActivityLogEntry;
import com.xebialabs.xlrelease.domain.Task;

import static com.xebialabs.deployit.checks.Checks.checkArgument;

@Component
public class TaskFieldsComparatorService {

    private static final Logger logger = LoggerFactory.getLogger(TaskFieldsComparatorService.class);

    private Map<Class<? extends Task>, TaskFieldsComparator<? extends Task>> taskFieldsComparatorsPerType;
    private DefaultTaskFieldsComparator defaultTaskFieldsComparator;

    @Autowired
    public TaskFieldsComparatorService(final List<? extends TaskFieldsComparator<? extends Task>> taskFieldsComparators,
                                       final DefaultTaskFieldsComparator defaultTaskFieldsComparator) {
        this.taskFieldsComparatorsPerType = mapTaskFieldsComparators(taskFieldsComparators);
        this.defaultTaskFieldsComparator = defaultTaskFieldsComparator;
    }

    public <T extends Task> List<ActivityLogEntry> getLogs(Date timestamp, String username, T original, T updated) {
        checkArgument(original.getClass().equals(updated.getClass()),
            "Cannot compare tasks of different type: %s and %s", original.getClass(), updated.getClass());

        if (taskFieldsComparatorsPerType.containsKey(original.getClass())) {
            @SuppressWarnings("unchecked")
            TaskFieldsComparator<T> comparator = (TaskFieldsComparator<T>) taskFieldsComparatorsPerType.get(original.getClass());
            return comparator.getLogs(timestamp, username, original, updated);
        }

        logger.debug(String.format("Using default task fields comparator for task type %s", original.getClass()));
        return defaultTaskFieldsComparator.getLogs(timestamp, username, original, updated);
    }

    private Map<Class<? extends Task>, TaskFieldsComparator<?>> mapTaskFieldsComparators(
        final List<? extends TaskFieldsComparator<? extends Task>> taskFieldsComparators) {

        Map<Class<? extends Task>, TaskFieldsComparator<?>> comparatorsPerType = new HashMap<>(taskFieldsComparators.size());
        taskFieldsComparators.forEach(comparator -> {
            Class<? extends Task> taskType = comparator.getTaskClass();
            if (comparatorsPerType.containsKey(taskType)) {
                throw new IllegalStateException(String.format("Two fields comparators are registered for the same task type %s: %s and %s",
                    taskType, comparatorsPerType.get(taskType), comparator));
            }
            comparatorsPerType.put(taskType, comparator);
        });
        return comparatorsPerType;
    }

}
