package com.xebialabs.xlrelease.activity

import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.domain.{ActivityLogEntry, ReleaseActivity}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.ActivityLogRepository
import com.xebialabs.xlrelease.variable.VariableHelper.withVariableSyntax
import org.springframework.stereotype.Component

import java.util.Date

@Component
@EventListener
class FolderVariableActivityLoggingEventHandler(val eventBus: XLReleaseEventBus, val activityLogRepository: ActivityLogRepository)
  extends ActivityLogger[FolderVariableEvent] {

  @Subscribe
  def onEvent(event: FolderVariableEvent): Unit = log(event)

  override def logEntries: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev@FolderVariableCreatedEvent(variable, source) =>
      LoggingParams(containerId(variable, source), None,
        ReleaseActivity.FOLDER_VARIABLE_CREATED.create(ev, ev.variable, withVariableSyntax(variable.getKey))
      )

    case ev@FolderVariableDeletedEvent(variable, source) =>
      LoggingParams(containerId(variable, source), None,
        ReleaseActivity.FOLDER_VARIABLE_DELETED.create(ev, ev.variable, withVariableSyntax(variable.getKey))
      )

    case ev@FolderVariableUpdatedEvent(original, updated, source) =>
      val logs = getDiffLogEntries(ev.timestamp, ev.username, original, updated)
      LoggingParams(containerId(updated, source), None, logEntries = logs)
  }

  private def containerId(variable: Variable, source: VariableModificationSource): Option[String] = {
    Option(source) match {
      case None => Option(variable.getFolderId)
      case Some(ModifiedFromScriptTask(taskId)) => releaseIdFrom(taskId)
    }
  }

  private def getDiffLogEntries(timestamp: Date, username: String, before: Variable, after: Variable): List[ActivityLogEntry] = {
    def diff(activityType: ReleaseActivity, key: String, value: Variable => String, showValues: Boolean = true): List[ActivityLogEntry] =
      if (value(before) == value(after)) {
        List.empty
      } else if (showValues) {
        List(activityType.create(timestamp, username, before.getType, before.getId, key, value(before), value(after)))
      } else {
        List(activityType.create(timestamp, username, before.getType, before.getId, key))
      }

    val oldKey: String = withVariableSyntax(before.getKey)
    val newKey: String = withVariableSyntax(after.getKey)

    val valueChangedActivity = if (before.isPassword) {
      diff(ReleaseActivity.FOLDER_VARIABLE_PASSWORD_VALUE_UPDATED, newKey,
        v => PasswordEncrypter.getInstance().ensureDecrypted(v.getValueAsString), showValues = false)
    } else {
      diff(ReleaseActivity.FOLDER_VARIABLE_VALUE_UPDATED, newKey, _.getValueAsString)
    }


    diff(ReleaseActivity.FOLDER_VARIABLE_NAME_UPDATED, oldKey, _.getKey) ++
      valueChangedActivity ++
      diff(ReleaseActivity.FOLDER_VARIABLE_LABEL_UPDATED, newKey, _.getLabel) ++
      diff(ReleaseActivity.FOLDER_VARIABLE_DESCRIPTION_UPDATED, newKey, _.getDescription)
  }

}
