package com.xebialabs.xlrelease.activity;

import java.util.Date;
import java.util.List;

import com.xebialabs.xlrelease.domain.ActivityLogEntry;
import com.xebialabs.xlrelease.domain.BaseConfiguration;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.service.UserInfoResolver;

import static com.google.common.base.Objects.equal;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.xlrelease.activity.ReleaseActivityDateFormatter.formatDate;
import static com.xebialabs.xlrelease.activity.ReleaseActivityUserFormatter.quoteUser;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_ABORT_RELEASE_ON_FAILURE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_ALLOW_PASSWORDS_IN_ALL_FIELDS_ON_FAILURE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_DESCRIPTION_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_DISABLE_NOTIFICATIONS_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_DUE_DATE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_FLAG_COMMENT_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_FLAG_STATUS_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_OWNER_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_RISK_PROFILE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_SCHEDULED_START_DATE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_TAGS_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.RELEASE_TITLE_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TEMPLATE_ALLOW_CONCURRENT_RELEASES_FROM_TRIGGER_UPDATED;
import static com.xebialabs.xlrelease.domain.ReleaseActivity.TEMPLATE_AUTHOR_UPDATED;
import static com.xebialabs.xlrelease.risk.domain.RiskProfile.RISK_PROFILE;

public class ReleaseFieldsComparator {

    private Date timestamp;

    private String username;

    private Release original;

    private Release updated;

    private UserInfoResolver userInfoResolver;

    public ReleaseFieldsComparator(Date timestamp, String username, Release original, Release updated, UserInfoResolver userInfoResolver) {
        this.timestamp = timestamp;
        this.username = username;
        this.original = original;
        this.updated = updated;
        this.userInfoResolver = userInfoResolver;
    }

    public List<ActivityLogEntry> getLogs() {
        List<ActivityLogEntry> logEntries = newArrayList();

        if (!equal(original.getTitle(), updated.getTitle())) {
            logEntries.add(RELEASE_TITLE_UPDATED.create(timestamp, username, original.getType(), original.getId(), original.getTitle(), updated.getTitle()));
        }

        if (!equal(original.getDescription(), updated.getDescription())) {
            logEntries.add(RELEASE_DESCRIPTION_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    original.getDescription(), updated.getDescription())
            );
        }

        if (!equal(original.getAuthor(), updated.getAuthor())) {
            logEntries.add(TEMPLATE_AUTHOR_UPDATED.create(timestamp, username, original.getType(), original.getId(), original.getAuthor(), updated.getAuthor()));
        }

        if (!equal(original.getDueDate(), updated.getDueDate())) {
            logEntries.add(RELEASE_DUE_DATE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    formatDate(original.getDueDate()), formatDate(updated.getDueDate()))
            );
        }

        if (!equal(original.getScheduledStartDate(), updated.getScheduledStartDate())) {
            logEntries.add(RELEASE_SCHEDULED_START_DATE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    formatDate(original.getScheduledStartDate()), formatDate(updated.getScheduledStartDate()))
            );
        }

        if (!equal(original.getOwner(), updated.getOwner())) {
            logEntries.add(RELEASE_OWNER_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    quoteUser(original.getOwner(), userInfoResolver), quoteUser(updated.getOwner(), userInfoResolver))
            );
        }

        // original.getTags() when set is returned as a ListOfStringView, therefore equal() won't work here.
        if (!equal(newArrayListOrNull(original.getTags()), updated.getTags())) {
            logEntries.add(RELEASE_TAGS_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    updated.getTags() != null ? newHashSet(updated.getTags()) : null)
            );
        }

        if (!equal(original.isAbortOnFailure(), updated.isAbortOnFailure())) {
            logEntries.add(RELEASE_ABORT_RELEASE_ON_FAILURE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    original.isAbortOnFailure(), updated.isAbortOnFailure())
            );
        }

        if (!equal(original.isAllowPasswordsInAllFields(), updated.isAllowPasswordsInAllFields())) {
            logEntries.add(RELEASE_ALLOW_PASSWORDS_IN_ALL_FIELDS_ON_FAILURE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    original.isAllowPasswordsInAllFields(), updated.isAllowPasswordsInAllFields())
            );
        }

        if (!equal(original.isDisableNotifications(), updated.isDisableNotifications())) {
            logEntries.add(RELEASE_DISABLE_NOTIFICATIONS_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    original.isDisableNotifications(), updated.isDisableNotifications())
            );
        }

        if (!original.isTemplate() && !equal(original.getFlagStatus(), updated.getFlagStatus())) {
            logEntries.add(RELEASE_FLAG_STATUS_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    original.getFlagStatus(), updated.getFlagStatus())
            );
        }

        if (!original.isTemplate() && !equal(original.getFlagComment(), updated.getFlagComment())) {
            logEntries.add(RELEASE_FLAG_COMMENT_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    original.getFlagComment(), updated.getFlagComment())
            );
        }

        if (original.isTemplate() && !equal(original.isAllowConcurrentReleasesFromTrigger(), updated.isAllowConcurrentReleasesFromTrigger())) {
            logEntries.add(TEMPLATE_ALLOW_CONCURRENT_RELEASES_FROM_TRIGGER_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    original.isAllowConcurrentReleasesFromTrigger(), updated.isAllowConcurrentReleasesFromTrigger())
            );
        }

        if (original.hasProperty(RISK_PROFILE) && !equal(original.getProperty(RISK_PROFILE), updated.getProperty(RISK_PROFILE))) {
            logEntries.add(RELEASE_RISK_PROFILE_UPDATED.create(timestamp, username, original.getType(), original.getId(),
                    getRiskProfileTitle(original.getProperty(RISK_PROFILE)), getRiskProfileTitle(updated.getProperty(RISK_PROFILE)))
            );
        }

        return logEntries;
    }

    private List<String> newArrayListOrNull(List<String> tags) {
        return tags != null ? newArrayList(tags) : null;
    }

    private String getRiskProfileTitle(BaseConfiguration riskProfile) {
        return null != riskProfile ? riskProfile.getTitle(): "";
    }

}
