package com.xebialabs.xlrelease.activity

import com.xebialabs.xlrelease.domain.ReleaseActivity._
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.{Dependency, PlanItem}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.ActivityLogRepository
import com.xebialabs.xlrelease.repository.Ids.{getName, isTaskId, phaseIdFrom}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@EventListener
class DependencyLoggingEventHandler @Autowired()(val eventBus: XLReleaseEventBus, val activityLogRepository: ActivityLogRepository)
  extends ActivityLogger[DependencyEvent] {

  @Subscribe
  def onEvent(event: DependencyEvent): Unit = log(event)

  override def logEntries: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev@DependencyCreatedEvent(dependency) =>
      LoggingParams(releaseIdFrom(dependency),
        None,
        DEPENDENCY_CREATED.create(ev, ev.dependency, dependency.getGateTask.getTitle, dependency.getTargetDisplayPath)
      )
    // don't log if isUpdateCausedByPhaseRestart
    case ev@DependencyUpdatedEvent(original, updated) if !isUpdateCausedByPhaseRestart(original, updated) =>
      LoggingParams(releaseIdFrom(updated),
        None,
        DEPENDENCY_UPDATED.create(ev, ev.original, updated.getGateTask.getTitle, original.getTargetDisplayPath, updated.getTargetDisplayPath)
      )
    case ev@DependencyDeletedEvent(dependency) =>
      LoggingParams(releaseIdFrom(dependency),
        None,
        DEPENDENCY_DELETED.create(ev, ev.dependency, dependency.getGateTask.getTitle, dependency.getTargetDisplayPath)
      )
  }

  /*
   * This supposes that a dependency was changed due to a phase restart if the target of the dependency
   * is a task whose ID only changed it's phase part, e.g.
   * Applications/Release1/PhaseOld/TaskParallel/Task123 to
   * Applications/Release1/PhaseNew/TaskParallel/Task123.
   */
  private def isUpdateCausedByPhaseRestart(original: Dependency, updated: Dependency): Boolean = {
    def targetTaskId(d: Dependency) = Option(d.getTarget[PlanItem])
      .map(_.getId)
      .filter(isTaskId)

    (targetTaskId(original), targetTaskId(updated)) match {
      case (Some(originalId), Some(updatedId)) =>
        val originalPhaseName = getName(phaseIdFrom(originalId))
        val updatedPhaseName = getName(phaseIdFrom(updatedId))
        val onlyPhaseHasChanged = originalPhaseName != updatedPhaseName &&
          originalId.replace(originalPhaseName, updatedPhaseName) == updatedId
        onlyPhaseHasChanged

      case _ => false
    }
  }

}
