package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.domain.ActivityLogEntry
import com.xebialabs.xlrelease.domain.ActivityLogEntry.BY_EVENT_TIME
import com.xebialabs.xlrelease.views.LogsFilters
import org.springframework.data.domain.Pageable

import java.util.{Date, List => JList}
import scala.jdk.CollectionConverters._

abstract class BaseActivityLogsService(val userInfoResolver: UserInfoResolver) {

  def matches(logEntry: ActivityLogEntry, filter: String): Boolean = {
    def matches(text: String): Boolean = {
      text != null && text.toLowerCase.contains(filter.toLowerCase)
    }

    (filter == null || matches(logEntry.getMessage)
      || matches(logEntry.getUsername)
      || matches(userInfoResolver.getFullNameOf(logEntry.getUsername)))
  }

  def between(logEntry: ActivityLogEntry, from: Date, to: Date): Boolean = {
    var predicate = false
    if (from == null && to == null) {
      predicate = true
    } else if (from == null) {
      predicate = logEntry.getEventTime.before(to) || logEntry.getEventTime.equals(to)
    } else if (to == null) {
      predicate = logEntry.getEventTime.after(from) || logEntry.getEventTime.equals(from)
    } else {
      predicate = (logEntry.getEventTime.after(from) || logEntry.getEventTime.equals(from)) &&
        (logEntry.getEventTime.before(to) || logEntry.getEventTime.equals(to))
    }
    predicate
  }

  def withTargetId(logEntry: ActivityLogEntry, targetId: String): Boolean = {
    var predicate = false
    if (targetId == null) {
      predicate = true
    } else {
      predicate = logEntry.getTargetId == targetId
    }
    predicate
  }

  def filterLogs(activityLogs: List[ActivityLogEntry], filters: LogsFilters, pageable: Pageable): List[ActivityLogEntry] = {
    val sortOrder = if (filters.isDateAsc) Ordering.comparatorToOrdering(BY_EVENT_TIME) else Ordering.comparatorToOrdering(BY_EVENT_TIME).reverse

    activityLogs
      .filter(logEntry => matches(logEntry, filters.getFilter))
      .filter(logEntry => between(logEntry, filters.getFrom, filters.getTo))
      .filter(logEntry => withCategories(logEntry, filters))
      .filter(logEntry => withTargetId(logEntry, filters.getTargetId))
      .sorted(sortOrder)
      .zipWithIndex
      .filter {
        case (e, index) => (pageable.isUnpaged) || ((index >= pageable.getOffset) && (index < pageable.getOffset + pageable.getPageSize))
      }
      .map(_._1)
  }

  def getFilteredLogs(containerId: String, filters: LogsFilters, pageable: Pageable): JList[ActivityLogEntry] = {
    fetchLogs(containerId, filters, pageable).asJava
  }

  def withCategories(logEntry: ActivityLogEntry, filters: LogsFilters): Boolean

  def fetchLogs(containerId: String, logsFilters: LogsFilters, pageable: Pageable): List[ActivityLogEntry]
}
