package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.domain.ActivityCategory._
import com.xebialabs.xlrelease.domain.{ActivityCategory, ActivityLogEntry, ReleaseActivity}
import com.xebialabs.xlrelease.repository.{ActivityLogRepository, ReleaseRepository}
import com.xebialabs.xlrelease.views.LogsFilters
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.Pageable
import org.springframework.stereotype.Service

import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

@Service
class ReleaseActivityLogsService @Autowired()(userInfoResolver: UserInfoResolver,
                                              val activityLogRepository: ActivityLogRepository,
                                              val releaseRepository: ReleaseRepository,
                                              val archivedActivityLogsService: ArchivedActivityLogsService
                                             ) extends BaseActivityLogsService(userInfoResolver) {
  private lazy val releaseActivityOps = ReleaseActivity.values().toList.asJava

  override def withCategories(logEntry: ActivityLogEntry, filters: LogsFilters): Boolean = {
    val allowedCategories = ListBuffer.empty[ActivityCategory]
    if (filters.withImportant) allowedCategories += IMPORTANT
    if (filters.withReleaseEdit) allowedCategories += RELEASE_EDIT
    if (filters.withTaskEdit) allowedCategories += TASK_EDIT
    if (filters.withComments) allowedCategories += COMMENTS
    if (filters.withLifecycle) allowedCategories += LIFECYCLE
    if (filters.withReassign) allowedCategories += REASSIGN
    if (filters.withReportingRecordEdit) allowedCategories += REPORTING_RECORD_EDIT
    if (filters.withSecurity) allowedCategories += SECURITY
    if (filters.withOther) allowedCategories += OTHER

    val activityCategories = ReleaseActivity.safeValueOf(logEntry.getActivityType).getCategories.asScala
    allowedCategories exists activityCategories
  }

  override def fetchLogs(containerId: String, logsFilters: LogsFilters, pageable: Pageable): List[ActivityLogEntry] = {
    if (releaseRepository.exists(containerId)) {
      activityLogRepository.findAllLogsOf(containerId, logsFilters, releaseActivityOps, pageable).asScala.toList
    } else if (archivedActivityLogsService.exists(containerId)) {
      filterLogs(
        archivedActivityLogsService.getActivityLogs(containerId).asScala.toList,
        logsFilters,
        pageable)
    } else {
      throw new NotFoundException(s"Release [${containerId}] does not exist in the repository or archive")
    }
  }
}
