package com.xebialabs.xlrelease.activity

import com.xebialabs.xlrelease.domain.ActivityLogEntry
import com.xebialabs.xlrelease.domain.events.{ActivityLogEvent, XLReleaseEvent}
import com.xebialabs.xlrelease.events.{EventListener, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.ActivityLogRepository
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

@EventListener
trait ActivityLogger[T <: XLReleaseEvent] extends Logging with ActivityLogEntryFactory {

  def eventBus: XLReleaseEventBus

  def activityLogRepository: ActivityLogRepository

  private val ignore: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case default =>
      logger.debug(s"Event $default not traced into activity logs")
      LoggingParams(None, None, List())
  }

  def log(event: XLReleaseEvent): Unit = {
    (logEntries orElse ignore).apply(event) match {
      case LoggingParams(Some(containerId), Some(user), entries) =>
        if (entries.nonEmpty) {
          logToInfo(entries)
          if (entries.lengthCompare(1) == 0) {
            activityLogRepository.log(containerId, entries.head, user)
          } else {
            activityLogRepository.log(containerId, entries.asJava, user)
          }
          entries.foreach(entry => eventBus.publish(
            ActivityLogEvent(containerId, entry.getId, entry.getActivityType, entry.getMessage, entry.getEventTime, entry.getTargetType, entry.getTargetId))
          )
        }
      case LoggingParams(Some(containerId), None, entries) =>
        if (entries.nonEmpty) {
          logToInfo(entries)
          if (entries.lengthCompare(1) == 0) {
            activityLogRepository.log(containerId, entries.head)
          } else {
            activityLogRepository.log(containerId, entries.asJava)
          }
          entries.foreach(entry => eventBus.publish(
            ActivityLogEvent(containerId, entry.getId, entry.getActivityType, entry.getMessage, entry.getEventTime, entry.getTargetType, entry.getTargetId))
          )
        }
      case LoggingParams(None, None, List()) =>
      // do nothing
      case _ =>
        logger.debug(s"Event $event not traced into activity logs")
    }
  }

  private def logToInfo(entries: List[ActivityLogEntry]): Unit = {
    entries.foreach(entry => entry.logEntryToInfo())
  }

}
