package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.core.xml.WriteWithoutPasswordCiConverter
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.domain.{ActivityLogEntry, ReleaseActivity}
import com.xebialabs.xlrelease.repository.ActivityLogRepository
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.serialization.json.xltype.CiListJson2Reader
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Service
class ArchivedActivityLogsService @Autowired()(
                                                activityLogRepository: ActivityLogRepository,
                                                archivedReleases: ArchivedReleases
                                              ) extends ArchivedActivityLogsSerializer {
  private lazy val releaseActivityOps = ReleaseActivity.values().toList.asJava

  @Timed
  def getActivityLogs(releaseId: String): JList[ActivityLogEntry] = {
    getActivityLogsOption(releaseId).getOrElse {
      throw new NotFoundException(s"Could not find activity logs for archived release [$releaseId]")
    }
  }

  private def getActivityLogsOption(releaseId: String): Option[JList[ActivityLogEntry]] = {
    archivedReleases.getActivityLogs(releaseId).map { activityLogsJson =>
      val ciConverter = new WriteWithoutPasswordCiConverter()

      val ciReader = CiListJson2Reader.create(activityLogsJson)
      ciConverter.readCis(ciReader).asInstanceOf[JList[ActivityLogEntry]]
    }
  }

  @Timed
  def exists(releaseId: String): Boolean = archivedReleases.exists(releaseId)

  @Timed
  override def serializeActivityLogsOf(releaseId: String): String = {
    serialize(activityLogRepository.findAllLogsOf(releaseId), new WriteWithoutPasswordCiConverter())
  }

}
